<?php

require_once __DIR__ . '/../../config/session.php';
require_once __DIR__ . '/../../config/auth.php';

if (!hasModuleAccess('logistica')) {
    http_response_code(403);
    header('Content-Type: application/json; charset=utf-8');
    echo json_encode(['error' => 'Acceso denegado', 'module' => 'logistica']);
    exit;
}

// 1. CONFIGURACIÓN
date_default_timezone_set('America/Lima');
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *'); 
header('X-Content-Type-Options: nosniff');

ini_set('display_errors', 0);
error_reporting(E_ALL);

define('BACKUP_FILE', 'data_backup.json');

require_once 'config.php';

// Validación de credenciales
if (!defined('NOTION_TOKEN') || !defined('NOTION_DB_ID') || !defined('NOTION_DB_PRODUCCION_ID')) {
    http_response_code(500);
    echo json_encode(['error' => 'Error Crítico: Faltan credenciales en config.php']); 
    exit;
}

// --- FUNCIÓN HELPER: CARGAR RESPALDO ---
function serveBackup($reason) {
    if (file_exists(BACKUP_FILE)) {
        $backupData = json_decode(file_get_contents(BACKUP_FILE), true);
        if ($backupData) {
            $backupData['meta']['source'] = 'backup_offline';
            $backupData['meta']['error_msg'] = $reason;
            // Registrar error en el log del servidor (invisible al usuario)
            error_log("[Panel Compipro] Sirviendo backup por fallo API: $reason");
            echo json_encode($backupData);
            exit;
        }
    }
    // Si no hay backup ni API
    http_response_code(503);
    echo json_encode(['error' => "Servicio no disponible y sin respaldo. Motivo: $reason"]);
    exit;
}

// --- FUNCIÓN MAESTRA: CONSULTA NOTION ---
function queryNotionDB_All($dbId) {
    $allResults = [];
    $hasMore = true;
    $nextCursor = null;

    while ($hasMore) {
        $curl = curl_init();
        $params = ["page_size" => 100];
        if ($nextCursor) $params["start_cursor"] = $nextCursor;
        
        curl_setopt_array($curl, [
            CURLOPT_URL => "https://api.notion.com/v1/databases/" . $dbId . "/query",
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_CUSTOMREQUEST => "POST",
            CURLOPT_POSTFIELDS => json_encode($params),
            CURLOPT_HTTPHEADER => [
                "Authorization: Bearer " . NOTION_TOKEN,
                "Notion-Version: 2022-06-28",
                "Content-Type: application/json"
            ],
            CURLOPT_TIMEOUT => 25, // Tiempo máximo de espera
            CURLOPT_CONNECTTIMEOUT => 10
        ]);

        $response = curl_exec($curl);
        $httpCode = curl_getinfo($curl, CURLINFO_HTTP_CODE);
        $curlErr = curl_error($curl);
        curl_close($curl);

        if ($curlErr) return ['error' => "Error Red: $curlErr"];
        if ($httpCode !== 200) {
            $errData = json_decode($response, true);
            return ['error' => "API $httpCode: " . ($errData['message'] ?? 'Desconocido')];
        }

        $data = json_decode($response, true);
        if (!$data || !isset($data['results'])) return ['error' => 'Respuesta JSON inválida'];

        $allResults = array_merge($allResults, $data['results']);
        $hasMore = $data['has_more'] ?? false;
        $nextCursor = $data['next_cursor'] ?? null;
    }
    return ['results' => $allResults];
}

// INICIO DEL PROCESO PRINCIPAL

// Estructura Base
$response = [
    'meta' => ['timestamp' => time(), 'source' => 'live'],
    'kpis' => ['activos' => 0, 'entregados' => 0, 'reprocesos' => 0, 'pendientes' => 0, 'avance_total' => 0],
    'status' => ['En curso'=>0, 'Entregado'=>0, 'Cerrado'=>0, 'Reproceso'=>0],       
    'weeks' => ['week1'=>[], 'week2'=>[], 'week3'=>[], 'week4'=>[], 'week5'=>[]],
    'criticality' => [], 
    'closing' => [
        'EXCELENTE'=>['count'=>0,'projects'=>[]], 
        'BUENO'=>['count'=>0,'projects'=>[]], 
        'REGULAR'=>['count'=>0,'projects'=>[]], 
        'PENDIENTE'=>['count'=>0,'projects'=>[]], 
        'REPROCESO'=>['count'=>0,'projects'=>[]]
    ],
    'has_reproceso' => false,
    'reproceso_projects' => []
];

// 1. Obtener Datos (Con protección Try/Catch por si el servidor falla)
try {
    $rawProyectos = queryNotionDB_All(NOTION_DB_ID);
    
    // Si la API falla, activamos el respaldo
    if (isset($rawProyectos['error'])) {
        serveBackup($rawProyectos['error']);
    }

} catch (Exception $e) {
    serveBackup($e->getMessage());
}

// 2. Procesar Proyectos
$totalAvance = 0;
$countProjects = 0;

if (!empty($rawProyectos['results'])) {
    foreach ($rawProyectos['results'] as $page) {
        $props = $page['properties'];
        
        // --- EXTRACCIÓN SEGURA DE DATOS ---
        
        // Nombre
        $fullName = 'Sin Nombre';
        if (!empty($props['Nombre']['title'])) {
            $fullName = $props['Nombre']['title'][0]['plain_text'];
        }
        // Acortar nombre para gráficas (UTF-8 seguro)
        $shortName = (mb_strlen($fullName) > 18) ? mb_substr($fullName, 0, 18) . '...' : $fullName;

        // Estado
        $statusRaw = 'En curso';
        if (isset($props['Progreso']['status']['name'])) $statusRaw = $props['Progreso']['status']['name'];
        elseif (isset($props['Progreso']['select']['name'])) $statusRaw = $props['Progreso']['select']['name'];
        
        $status = 'En curso'; 
        if (mb_stripos($statusRaw, 'Entregado') !== false || mb_stripos($statusRaw, 'Aprobado') !== false) $status = 'Entregado';
        elseif (mb_stripos($statusRaw, 'Cerrado') !== false) $status = 'Cerrado';
        elseif (mb_stripos($statusRaw, 'Reproceso') !== false) $status = 'Reproceso';
        
        if (!isset($response['status'][$status])) $response['status'][$status] = 0;
        $response['status'][$status]++;

        // Lógica Reproceso (Alerta)
        $isReproceso = ($status === 'Reproceso');
        if ($isReproceso) {
            $response['has_reproceso'] = true;
            $response['reproceso_projects'][] = [
                'name' => $fullName,
                'criticality' => 'Verificar Estado'
            ];
        }

        // Criticidad Logística
        $critLogistica = 'NORMAL';
        if (isset($props['Criticidad Logistica']['select']['name'])) {
            $critLogistica = mb_strtoupper(trim($props['Criticidad Logistica']['select']['name']));
        }

        // Cierre
        $closingRaw = isset($props['Tipo de cierre']['select']['name']) ? mb_strtoupper(trim($props['Tipo de cierre']['select']['name'])) : 'PENDIENTE';
        $closing = 'PENDIENTE';
        if (strpos($closingRaw, 'EXCEL') !== false) $closing = 'EXCELENTE';
        elseif (strpos($closingRaw, 'BUEN') !== false) $closing = 'BUENO';
        elseif (strpos($closingRaw, 'REGULAR') !== false) $closing = 'REGULAR';
        elseif (strpos($closingRaw, 'REPROCESO') !== false) $closing = 'REPROCESO';
        
        if (!isset($response['closing'][$closing])) $response['closing'][$closing] = ['count'=>0, 'projects'=>[]];
        $response['closing'][$closing]['count']++;
        $response['closing'][$closing]['projects'][] = $fullName;

        // Avance
        $avanceRaw = $props['API_Avance']['formula']['number'] ?? null;
        if ($avanceRaw === null) {
            $avanceRaw = $props['Avance']['number'] ?? ($props['# Avance']['number'] ?? 0);
        }
        $avance = ($avanceRaw <= 1 && $avanceRaw > 0) ? $avanceRaw * 100 : $avanceRaw;
        $avance = round((float)$avance);

        // Semanas (Fecha Inicio)
        $dateStr = $props['Periodo']['date']['start'] ?? date('Y-m-d');
        $projTimestamp = strtotime($dateStr);
        
        if (date('Y-m', $projTimestamp) === date('Y-m')) {
            $day = date('j', $projTimestamp);
            $weekNum = ceil($day / 7);
            if($weekNum > 5) $weekNum = 5; if($weekNum < 1) $weekNum = 1;
            
            $response['weeks']['week'.$weekNum][] = [
                'name' => $shortName,      
                'fullName' => $fullName,   
                'progress' => $avance,
                'date' => $dateStr,      
                'is_reproceso' => $isReproceso, 
                'logistica' => $critLogistica   
            ];
        }

        $totalAvance += $avance;
        $countProjects++;
    }
}

// Liberar memoria del array crudo grande
unset($rawProyectos);

// 3. Procesar Producción
$rawProduccion = queryNotionDB_All(NOTION_DB_PRODUCCION_ID);
if (!isset($rawProduccion['error']) && !empty($rawProduccion['results'])) {
    foreach ($rawProduccion['results'] as $page) {
        $props = $page['properties'];
        $prodName = !empty($props['Producto']['title']) ? $props['Producto']['title'][0]['plain_text'] : 'Item';
        
        $urgenciaRaw = $props['Urgencia']['select']['name'] ?? 'Sin asignar';
        $urgencia = ucfirst(mb_strtolower($urgenciaRaw)); 
        
        if (!isset($response['criticality'][$urgencia])) {
            $response['criticality'][$urgencia] = ['count' => 0, 'projects' => []];
        }
        $response['criticality'][$urgencia]['count']++;
        $response['criticality'][$urgencia]['projects'][] = $prodName;
    }
}
unset($rawProduccion);

// 4. Calcular KPIs
$response['kpis']['activos'] = $response['status']['En curso'] ?? 0;
$response['kpis']['entregados'] = $response['status']['Entregado'] ?? 0;
$response['kpis']['reprocesos'] = $response['status']['Reproceso'] ?? 0;
$response['kpis']['pendientes'] = ($response['closing']['PENDIENTE']['count'] ?? 0) + 
                                  ($response['closing']['REPROCESO']['count'] ?? 0);
$response['kpis']['avance_total'] = $countProjects > 0 ? round($totalAvance / $countProjects) : 0;

// 5. Filtrado Top 5 + Urgentes
foreach ($response['weeks'] as $k => $projects) {
    // A. Filtro Estricto: 'URGENTE'
    $filteredProjects = array_filter($projects, function($p) {
        return (strpos($p['logistica'], 'URGENTE') !== false);
    });
    // B. Top 5
    $response['weeks'][$k] = array_slice(array_values($filteredProjects), 0, 5);
}

// =================================================================
// 6. ESCRITURA ATÓMICA DE RESPALDO (SEGURIDAD DE ARCHIVOS)
// =================================================================

// Usamos un archivo temporal y luego renombramos para evitar que se lea un archivo a medio escribir
$tempFile = BACKUP_FILE . '.tmp';
if (file_put_contents($tempFile, json_encode($response))) {
    rename($tempFile, BACKUP_FILE);
}

// Salida Final
echo json_encode($response, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
?>