<?php

class MauticApiClient
{
    private string $baseUrl;
    private string $authType;
    private string $apiUser;
    private string $apiPass;
    private string $accessToken;

    public function __construct(string $baseUrl, array $options)
    {
        $this->baseUrl = rtrim($baseUrl, '/');
        $this->authType = $options['auth_type'] ?? 'basic';
        $this->apiUser = $options['api_user'] ?? '';
        $this->apiPass = $options['api_pass'] ?? '';
        $this->accessToken = $options['access_token'] ?? '';
    }

    public function get(string $path, array $query = []): array
    {
        return $this->request('GET', $path, null, $query);
    }

    public function post(string $path, array $payload): array
    {
        return $this->request('POST', $path, $payload);
    }

    public function patch(string $path, array $payload): array
    {
        return $this->request('PATCH', $path, $payload);
    }

    private function request(string $method, string $path, ?array $payload = null, array $query = []): array
    {
        $url = $this->baseUrl . '/api/' . ltrim($path, '/');
        if (!empty($query)) {
            $url .= '?' . http_build_query($query);
        }

        $ch = curl_init($url);
        $headers = [
            'Accept: application/json',
        ];

        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_CUSTOMREQUEST, $method);
        if ($this->authType === 'oauth2') {
            $headers[] = 'Authorization: Bearer ' . $this->accessToken;
        } else {
            curl_setopt($ch, CURLOPT_USERPWD, $this->apiUser . ':' . $this->apiPass);
            curl_setopt($ch, CURLOPT_HTTPAUTH, CURLAUTH_BASIC);
        }
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);

        if ($payload !== null) {
            $headers[] = 'Content-Type: application/json';
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($payload));
        }

        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);

        if ($response === false) {
            $error = curl_error($ch);
            curl_close($ch);
            throw new RuntimeException('Error en API Mautic: ' . $error);
        }
        curl_close($ch);

        $decoded = json_decode($response, true);
        if ($httpCode >= 400) {
            $message = $decoded['error']['message'] ?? $decoded['message'] ?? $response;
            throw new RuntimeException('Respuesta API Mautic ' . $httpCode . ': ' . $message);
        }

        return is_array($decoded) ? $decoded : [];
    }
}

class MauticSync
{
    private PDO $db;

    public function __construct(PDO $db)
    {
        $this->db = $db;
    }

    public function syncAll(int $contactLimit = 0): array
    {
        $instances = $this->fetchInstances();
        $results = [];

        foreach ($instances as $instance) {
            $results[] = $this->syncInstance($instance, $contactLimit);
        }

        return [
            'success' => true,
            'instances' => $results,
        ];
    }

    public function syncContactsOnly(int $contactLimit = 0): array
    {
        $instances = $this->fetchInstances();
        $results = [];

        foreach ($instances as $instance) {
            $client = $this->buildClient($instance);
            $segmentId = $this->getSegmentId((int)$instance['id'], 'contactcrm');
            $results[] = [
                'instancia_id' => (int)$instance['id'],
                'push' => $this->pushContacts(
                    $client,
                    (int)$instance['id'],
                    $segmentId,
                    $instance['last_sync_at'],
                    $contactLimit
                ),
            ];
            if (!empty($results[count($results) - 1]['push']['completed'])) {
                $this->updateInstanceSync((int)$instance['id']);
            }
        }

        return [
            'success' => true,
            'instances' => $results,
        ];
    }

    public function syncCampaignsOnly(): array
    {
        $instances = $this->fetchInstances();
        $results = [];

        foreach ($instances as $instance) {
            $client = $this->buildClient($instance);
            $minCampaignId = isset($instance['min_campaign_id']) ? (int)$instance['min_campaign_id'] : 0;
            $results[] = [
                'instancia_id' => (int)$instance['id'],
                'pull' => $this->pullCampaigns($client, (int)$instance['id'], $minCampaignId, false),
            ];
        }

        return [
            'success' => true,
            'instances' => $results,
        ];
    }

    private function syncInstance(array $instance, int $contactLimit = 0): array
    {
        $client = $this->buildClient($instance);
        $segmentId = $this->getSegmentId((int)$instance['id'], 'contactcrm');
        $minCampaignId = isset($instance['min_campaign_id']) ? (int)$instance['min_campaign_id'] : 0;

        $pushSummary = $this->pushContacts(
            $client,
            (int)$instance['id'],
            $segmentId,
            $instance['last_sync_at'],
            $contactLimit
        );
        $pullSummary = $this->pullCampaigns(
            $client,
            (int)$instance['id'],
            $minCampaignId,
            true,
            $instance['last_report_sync_at'] ?? null
        );

        if (!empty($pushSummary['completed'])) {
            $this->updateInstanceSync((int)$instance['id']);
        }

        return [
            'instancia_id' => (int)$instance['id'],
            'push' => $pushSummary,
            'pull' => $pullSummary,
        ];
    }

    private function fetchInstances(): array
    {
        $stmt = $this->db->query(
            "SELECT id, base_url, auth_type, api_user, api_pass,
                    oauth_client_id, oauth_client_secret, oauth_redirect_uri,
                    oauth_access_token, oauth_refresh_token, oauth_expires_at,
                    last_sync_at, last_report_sync_at, min_campaign_id
             FROM mautic_instancias
             WHERE activo = 1"
        );
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    private function getSegmentId(int $instanceId, string $segmentKey): ?int
    {
        $stmt = $this->db->prepare(
            "SELECT mautic_segment_id
             FROM mautic_segmentos_map
             WHERE mautic_instancia_id = :instancia_id AND segmento_clave = :clave
             LIMIT 1"
        );
        $stmt->execute([
            ':instancia_id' => $instanceId,
            ':clave' => $segmentKey,
        ]);
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        return $row ? (int)$row['mautic_segment_id'] : null;
    }

    private function pushContacts(
        MauticApiClient $client,
        int $instanceId,
        ?int $segmentId,
        ?string $lastSync,
        int $limit = 0
    ): array
    {
        $fetch = $this->fetchContacts($instanceId, $lastSync, $limit);
        $contacts = $fetch['rows'];
        $created = 0;
        $updated = 0;
        $skipped = 0;

        foreach ($contacts as $contact) {
            if (!$contact['email']) {
                $skipped++;
                continue;
            }

            $map = $this->findContactMap($instanceId, (int)$contact['id']);
            $payload = [
                'firstname' => $contact['nombres'],
                'lastname' => $contact['apellidos'] ?? '',
                'email' => $contact['email'],
                'phone' => $contact['celular'] ?? '',
                'company' => $contact['empresa'] ?? '',
            ];

            if ($map) {
                $client->patch('contacts/' . $map['mautic_lead_id'] . '/edit', $payload);
                $updated++;
                $leadId = (int)$map['mautic_lead_id'];
            } else {
                $response = $client->post('contacts/new', $payload);
                $leadId = (int)($response['contact']['id'] ?? 0);
                if ($leadId > 0) {
                    $this->insertContactMap($instanceId, (int)$contact['id'], $leadId);
                    $created++;
                } else {
                    $skipped++;
                    continue;
                }
            }

            if ($segmentId) {
                $client->post('segments/' . $segmentId . '/contact/' . $leadId . '/add', []);
            }

            $this->touchContactMap($instanceId, (int)$contact['id']);
        }

        return [
            'total' => count($contacts),
            'created' => $created,
            'updated' => $updated,
            'skipped' => $skipped,
            'completed' => !$fetch['has_more'],
        ];
    }

    private function fetchContacts(int $instanceId, ?string $lastSync, int $limit = 0): array
    {
        $sql = "SELECT c.id, c.nombres, c.apellidos, c.celular,
                       e.razon_social AS empresa,
                       ce.email
                FROM contacto c
                LEFT JOIN empresa e ON e.id = c.empresa_id
                LEFT JOIN contacto_emails ce
                    ON ce.contacto_id = c.id AND ce.estado = 1 AND ce.es_principal = 1";

        $params = [];
        $where = [];
        if (empty($lastSync)) {
            $sql .= " LEFT JOIN mautic_contacto_map m
                       ON m.contacto_id = c.id AND m.mautic_instancia_id = :instancia_id";
            $where[] = "m.id IS NULL";
            $params[':instancia_id'] = $instanceId;
        } else {
            $where[] = "(c.updated_at >= :last_sync OR c.created_at >= :last_sync OR ce.updated_at >= :last_sync)";
            $params[':last_sync'] = $lastSync;
        }
        if ($where) {
            $sql .= " WHERE " . implode(' AND ', $where);
        }
        $sql .= " ORDER BY c.id ASC";

        $hasMore = false;
        $limitPlus = 0;
        if ($limit > 0) {
            $limitPlus = $limit + 1;
            $sql .= " LIMIT :limit_plus";
        }

        $stmt = $this->db->prepare($sql);
        foreach ($params as $key => $value) {
            $stmt->bindValue($key, $value);
        }
        if ($limitPlus > 0) {
            $stmt->bindValue(':limit_plus', $limitPlus, PDO::PARAM_INT);
        }
        $stmt->execute();
        $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);

        if ($limit > 0 && count($rows) > $limit) {
            $hasMore = true;
            $rows = array_slice($rows, 0, $limit);
        }

        return ['rows' => $rows, 'has_more' => $hasMore];
    }

    private function findContactMap(int $instanceId, int $contactId): ?array
    {
        $stmt = $this->db->prepare(
            "SELECT id, mautic_lead_id
             FROM mautic_contacto_map
             WHERE mautic_instancia_id = :instancia_id AND contacto_id = :contacto_id
             LIMIT 1"
        );
        $stmt->execute([
            ':instancia_id' => $instanceId,
            ':contacto_id' => $contactId,
        ]);
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        return $row ?: null;
    }

    private function insertContactMap(int $instanceId, int $contactId, int $leadId): void
    {
        $stmt = $this->db->prepare(
            "INSERT INTO mautic_contacto_map (mautic_instancia_id, contacto_id, mautic_lead_id, last_sync_at)
             VALUES (:instancia_id, :contacto_id, :lead_id, NOW())"
        );
        $stmt->execute([
            ':instancia_id' => $instanceId,
            ':contacto_id' => $contactId,
            ':lead_id' => $leadId,
        ]);
    }

    private function touchContactMap(int $instanceId, int $contactId): void
    {
        $stmt = $this->db->prepare(
            "UPDATE mautic_contacto_map
             SET last_sync_at = NOW()
             WHERE mautic_instancia_id = :instancia_id AND contacto_id = :contacto_id"
        );
        $stmt->execute([
            ':instancia_id' => $instanceId,
            ':contacto_id' => $contactId,
        ]);
    }

    private function pullCampaigns(
        MauticApiClient $client,
        int $instanceId,
        int $minCampaignId = 0,
        bool $withEvents = true,
        ?string $lastReportSyncAt = null
    ): array
    {
        $campaigns = $this->getPaginated($client, 'campaigns', 'campaigns');
        $synced = 0;
        $contactsSynced = 0;
        $eventsSynced = 0;
        $reportConfig = $withEvents ? $this->getReportConfig($instanceId, 'all_emails') : null;

        foreach ($campaigns as $campaign) {
            if ($minCampaignId > 0 && (int)($campaign['id'] ?? 0) < $minCampaignId) {
                continue;
            }
            $campaniaId = $this->upsertCampaign($instanceId, $campaign);
            if (!$campaniaId) {
                continue;
            }
            $synced++;
            $contactsSynced += $this->syncCampaignContacts($client, $instanceId, $campaniaId, (int)$campaign['id']);
            if ($withEvents && !$reportConfig && $this->isCampaignActive($campaign)) {
                $eventsSynced += $this->syncCampaignEvents($client, $instanceId, $campaniaId);
            }
        }

        $campanias = $this->fetchCampaniasMap($instanceId);

        if ($withEvents && $reportConfig) {
            try {
                $reportDef = $this->getReportDefinition($client, (int)$reportConfig['report_id']);

                $historyCampaigns = $this->getHistoricalCampaignsFromMap($campanias);
                foreach ($historyCampaigns as $history) {
                    if ($this->hasCampaignEvents($instanceId, (int)$history['campania_id'])) {
                        continue;
                    }
                    $this->applyReportCampaignFilters($client, $reportDef, $reportConfig, $history);
                    $eventsSynced += $this->syncReportEvents(
                        $client,
                        $instanceId,
                        $reportConfig,
                        null,
                        (int)$history['mautic_campaign_id'],
                        null,
                        true,
                        null
                    );
                }

                $activeCampaigns = $this->getActiveCampaignsFromMap($campanias);
                foreach ($activeCampaigns as $active) {
                    $this->applyReportCampaignFilters($client, $reportDef, $reportConfig, $active);
                    $eventsSynced += $this->syncReportEvents(
                        $client,
                        $instanceId,
                        $reportConfig,
                        null,
                        (int)$active['mautic_campaign_id'],
                        null,
                        false,
                        null
                    );
                }
            } catch (Throwable $e) {
                foreach ($campaigns as $campaign) {
                    if ($this->isCampaignActive($campaign)) {
                        $campaniaId = $this->findCampaignIdByMautic($instanceId, (int)$campaign['id']);
                        if ($campaniaId) {
                            $eventsSynced += $this->syncCampaignEvents($client, $instanceId, $campaniaId);
                        }
                    }
                }
            }
        }

        return [
            'campaigns' => $synced,
            'contacts' => $contactsSynced,
            'events' => $eventsSynced,
        ];
    }

    private function upsertCampaign(int $instanceId, array $campaign): ?int
    {
        $map = $this->findCampaignMap($instanceId, (int)$campaign['id']);
        $payload = [
            'titulo' => $campaign['name'] ?? 'Campana',
            'asunto' => $campaign['description'] ?? null,
            'fecha_inicio' => $this->toDate($campaign['publishUp'] ?? null),
            'fecha_fin' => $this->toDate($campaign['publishDown'] ?? null),
        ];

        if ($map) {
            $stmt = $this->db->prepare(
                "UPDATE campanias
                 SET titulo = :titulo, asunto = :asunto, fecha_inicio = :fecha_inicio, fecha_fin = :fecha_fin,
                     mautic_instancia_id = :instancia_id, updated_at = NOW()
                 WHERE id_campania = :campania_id"
            );
            $stmt->execute([
                ':titulo' => $payload['titulo'],
                ':asunto' => $payload['asunto'],
                ':fecha_inicio' => $payload['fecha_inicio'],
                ':fecha_fin' => $payload['fecha_fin'],
                ':instancia_id' => $instanceId,
                ':campania_id' => $map['campania_id'],
            ]);
            $this->touchCampaignMap($instanceId, (int)$campaign['id']);
            return (int)$map['campania_id'];
        }

        $stmt = $this->db->prepare(
            "INSERT INTO campanias (mautic_instancia_id, titulo, asunto, fecha_inicio, fecha_fin, created_at)
             VALUES (:instancia_id, :titulo, :asunto, :fecha_inicio, :fecha_fin, NOW())"
        );
        $stmt->execute([
            ':instancia_id' => $instanceId,
            ':titulo' => $payload['titulo'],
            ':asunto' => $payload['asunto'],
            ':fecha_inicio' => $payload['fecha_inicio'],
            ':fecha_fin' => $payload['fecha_fin'],
        ]);
        $campaniaId = (int)$this->db->lastInsertId();
        $this->insertCampaignMap($instanceId, $campaniaId, (int)$campaign['id']);
        return $campaniaId;
    }

    private function findCampaignMap(int $instanceId, int $mauticCampaignId): ?array
    {
        $stmt = $this->db->prepare(
            "SELECT id, campania_id
             FROM mautic_campania_map
             WHERE mautic_instancia_id = :instancia_id AND mautic_campaign_id = :campaign_id
             LIMIT 1"
        );
        $stmt->execute([
            ':instancia_id' => $instanceId,
            ':campaign_id' => $mauticCampaignId,
        ]);
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        return $row ?: null;
    }

    private function findCampaignIdByMautic(int $instanceId, int $mauticCampaignId): ?int
    {
        $map = $this->findCampaignMap($instanceId, $mauticCampaignId);
        return $map ? (int)$map['campania_id'] : null;
    }

    private function insertCampaignMap(int $instanceId, int $campaniaId, int $mauticCampaignId): void
    {
        $stmt = $this->db->prepare(
            "INSERT INTO mautic_campania_map (mautic_instancia_id, campania_id, mautic_campaign_id, last_sync_at)
             VALUES (:instancia_id, :campania_id, :campaign_id, NOW())"
        );
        $stmt->execute([
            ':instancia_id' => $instanceId,
            ':campania_id' => $campaniaId,
            ':campaign_id' => $mauticCampaignId,
        ]);
    }

    private function touchCampaignMap(int $instanceId, int $mauticCampaignId): void
    {
        $stmt = $this->db->prepare(
            "UPDATE mautic_campania_map
             SET last_sync_at = NOW()
             WHERE mautic_instancia_id = :instancia_id AND mautic_campaign_id = :campaign_id"
        );
        $stmt->execute([
            ':instancia_id' => $instanceId,
            ':campaign_id' => $mauticCampaignId,
        ]);
    }

    private function syncCampaignContacts(MauticApiClient $client, int $instanceId, int $campaniaId, int $mauticCampaignId): int
    {
        $contacts = $this->getPaginated($client, 'campaigns/' . $mauticCampaignId . '/contacts', 'contacts');
        $count = 0;

        foreach ($contacts as $contact) {
            $leadId = $contact['id'] ?? $contact['leadId'] ?? $contact['lead_id'] ?? null;
            $email = $contact['email'] ?? $contact['emailAddress'] ?? $contact['email_address'] ?? null;
            if (!$email) {
                continue;
            }
            $contactoEmailId = $this->findContactoEmailId($email);
            if (!$contactoEmailId) {
                $this->recordPendingEmail($instanceId, $email, $mauticCampaignId, $campaniaId, $leadId ? (int)$leadId : null, 'campaign');
                continue;
            }
            $contactId = $this->findContactoIdByEmailId($contactoEmailId);
            if (!$contactId) {
                $this->recordPendingEmail($instanceId, $email, $mauticCampaignId, $campaniaId, $leadId ? (int)$leadId : null, 'campaign');
                continue;
            }
            if ($leadId) {
                $this->ensureContactMap($instanceId, (int)$contactId, (int)$leadId);
                $this->ensureContactoEmailMap($instanceId, (int)$contactoEmailId, (int)$leadId);
            }
            $stmt = $this->db->prepare(
                "INSERT IGNORE INTO campania_contacto (campania_id, contacto_id, created_at)
                 VALUES (:campania_id, :contacto_id, NOW())"
            );
            $stmt->execute([
                ':campania_id' => $campaniaId,
                ':contacto_id' => $contactId,
            ]);
            $count++;
        }

        return $count;
    }

    private function syncCampaignEvents(MauticApiClient $client, int $instanceId, int $campaniaId): int
    {
        $openSegment = $this->getSegmentId($instanceId, 'contactopenmail');
        $clickSegment = $this->getSegmentId($instanceId, 'contactclickmail');

        $events = 0;
        if ($openSegment) {
            $events += $this->syncSegmentContacts($client, $instanceId, $campaniaId, $openSegment, 'open');
        }
        if ($clickSegment) {
            $events += $this->syncSegmentContacts($client, $instanceId, $campaniaId, $clickSegment, 'click');
        }

        return $events;
    }

    private function syncSegmentContacts(
        MauticApiClient $client,
        int $instanceId,
        int $campaniaId,
        int $segmentId,
        string $tipo
    ): int {
        try {
            $contacts = $this->getPaginated($client, 'segments/' . $segmentId . '/contacts', 'contacts');
        } catch (Throwable $e) {
            return 0;
        }
        $count = 0;

        foreach ($contacts as $contact) {
            $email = $contact['email'] ?? $contact['emailAddress'] ?? $contact['email_address'] ?? null;
            if (!$email) {
                continue;
            }
            $contactoEmailId = $this->findContactoEmailId($email);
            if (!$contactoEmailId) {
                $this->recordPendingEmail($instanceId, $email, null, $campaniaId, (int)$contact['id'], 'segment');
                continue;
            }
            $contactoId = $this->findContactoIdByEmailId($contactoEmailId);
            if (!$contactoId) {
                $this->recordPendingEmail($instanceId, $email, null, $campaniaId, (int)$contact['id'], 'segment');
                continue;
            }
            $this->ensureContactoEmailMap($instanceId, (int)$contactoEmailId, (int)$contact['id']);

            $eventDate = $this->extractEventDate($contact);
            $this->upsertContactEvent($instanceId, $campaniaId, $contactoId, $tipo, $eventDate);
            $count++;
        }

        return $count;
    }

    private function extractEventDate(array $contact): string
    {
        $date = $contact['dateAdded'] ?? $contact['date_added'] ?? null;
        if ($date) {
            return str_replace('T', ' ', substr($date, 0, 19));
        }
        return date('Y-m-d H:i:s');
    }

    private function upsertContactEvent(
        int $instanceId,
        int $campaniaId,
        int $contactoId,
        string $tipo,
        string $eventDate
    ): void {
        $stmt = $this->db->prepare(
            "SELECT id, open, clicks
             FROM campania_contacto_eventos
             WHERE mautic_instancia_id = :instancia_id
               AND campania_id = :campania_id
               AND contacto_id = :contacto_id
             LIMIT 1"
        );
        $stmt->execute([
            ':instancia_id' => $instanceId,
            ':campania_id' => $campaniaId,
            ':contacto_id' => $contactoId,
        ]);
        $row = $stmt->fetch(PDO::FETCH_ASSOC);

        $opens = $row && ($row['open'] ?? 'No') === 'Yes' ? 'Yes' : 'No';
        $clicks = $row ? (int)$row['clicks'] : 0;

        if ($tipo === 'open') {
            $opens = 'Yes';
        } elseif ($tipo === 'click') {
            $clicks = max($clicks, 1);
            $opens = 'Yes';
        }

        if ($row) {
            $update = $this->db->prepare(
                "UPDATE campania_contacto_eventos
                 SET open = :open,
                     clicks = :clicks,
                     last_event_at = :last_event_at
                 WHERE id = :id"
            );
            $update->execute([
                ':open' => $opens,
                ':clicks' => $clicks,
                ':last_event_at' => $eventDate,
                ':id' => $row['id'],
            ]);
            return;
        }

        $insert = $this->db->prepare(
            "INSERT INTO campania_contacto_eventos
                (campania_id, contacto_id, mautic_instancia_id, open, clicks, last_event_at, created_at)
             VALUES
                (:campania_id, :contacto_id, :instancia_id, :open, :clicks, :last_event_at, NOW())"
        );
        $insert->execute([
            ':campania_id' => $campaniaId,
            ':contacto_id' => $contactoId,
            ':instancia_id' => $instanceId,
            ':open' => $opens,
            ':clicks' => $clicks,
            ':last_event_at' => $eventDate,
        ]);
    }

    private function findContactoIdByLead(int $instanceId, int $leadId): ?int
    {
        $stmt = $this->db->prepare(
            "SELECT contacto_id
             FROM mautic_contacto_map
             WHERE mautic_instancia_id = :instancia_id AND mautic_lead_id = :lead_id
             LIMIT 1"
        );
        $stmt->execute([
            ':instancia_id' => $instanceId,
            ':lead_id' => $leadId,
        ]);
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        return $row ? (int)$row['contacto_id'] : null;
    }

    private function getPaginated(MauticApiClient $client, string $path, string $key, int $limit = 100): array
    {
        $start = 0;
        $items = [];

        do {
            $response = $client->get($path, ['start' => $start, 'limit' => $limit]);
            $chunk = array_values($response[$key] ?? []);
            $items = array_merge($items, $chunk);
            $total = (int)($response['total'] ?? count($chunk));
            $start += $limit;
        } while ($start < $total);

        return $items;
    }

    private function toDate(?string $value): ?string
    {
        if (!$value) {
            return null;
        }
        return substr($value, 0, 10);
    }

    private function updateInstanceSync(int $instanceId): void
    {
        $stmt = $this->db->prepare(
            "UPDATE mautic_instancias
             SET last_sync_at = NOW()
             WHERE id = :id"
        );
        $stmt->execute([':id' => $instanceId]);
    }

    private function updateInstanceReportSync(int $instanceId, string $lastSentAt): void
    {
        $stmt = $this->db->prepare(
            "UPDATE mautic_instancias
             SET last_report_sync_at = :last_report_sync_at
             WHERE id = :id"
        );
        $stmt->execute([
            ':last_report_sync_at' => $lastSentAt,
            ':id' => $instanceId,
        ]);
    }

    private function buildClient(array $instance): MauticApiClient
    {
        $authType = $instance['auth_type'] ?? 'basic';
        if ($authType === 'oauth2') {
            $accessToken = $this->ensureAccessToken($instance);
            return new MauticApiClient($instance['base_url'], [
                'auth_type' => 'oauth2',
                'access_token' => $accessToken,
            ]);
        }

        return new MauticApiClient($instance['base_url'], [
            'auth_type' => 'basic',
            'api_user' => $instance['api_user'] ?? '',
            'api_pass' => $instance['api_pass'] ?? '',
        ]);
    }

    private function ensureAccessToken(array $instance): string
    {
        $expiresAt = $instance['oauth_expires_at'] ?? null;
        $accessToken = $instance['oauth_access_token'] ?? '';

        if ($accessToken !== '' && $expiresAt) {
            $now = new DateTime();
            $expiry = new DateTime($expiresAt);
            if ($now < $expiry) {
                return $accessToken;
            }
        }

        return $this->refreshToken($instance);
    }

    private function refreshToken(array $instance): string
    {
        $baseUrl = rtrim((string)$instance['base_url'], '/');
        $tokenUrl = $baseUrl . '/oauth/v2/token';

        $payload = [
            'grant_type' => 'refresh_token',
            'client_id' => $instance['oauth_client_id'] ?? '',
            'client_secret' => $instance['oauth_client_secret'] ?? '',
            'refresh_token' => $instance['oauth_refresh_token'] ?? '',
            'redirect_uri' => $instance['oauth_redirect_uri'] ?? '',
        ];

        $ch = curl_init($tokenUrl);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($payload));
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        $response = curl_exec($ch);
        if ($response === false) {
            $error = curl_error($ch);
            curl_close($ch);
            throw new RuntimeException('Error OAuth2: ' . $error);
        }
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        $decoded = json_decode($response, true);
        if ($httpCode >= 400 || !is_array($decoded) || empty($decoded['access_token'])) {
            $message = $decoded['error_description'] ?? $decoded['error'] ?? $response;
            throw new RuntimeException('OAuth2 refresh error: ' . $message);
        }

        $accessToken = $decoded['access_token'];
        $refreshToken = $decoded['refresh_token'] ?? ($instance['oauth_refresh_token'] ?? '');
        $expiresIn = isset($decoded['expires_in']) ? (int)$decoded['expires_in'] : 3600;
        $expiresAt = (new DateTime())->add(new DateInterval('PT' . $expiresIn . 'S'))->format('Y-m-d H:i:s');

        $stmt = $this->db->prepare(
            "UPDATE mautic_instancias
             SET oauth_access_token = :access_token,
                 oauth_refresh_token = :refresh_token,
                 oauth_expires_at = :expires_at
             WHERE id = :id"
        );
        $stmt->execute([
            ':access_token' => $accessToken,
            ':refresh_token' => $refreshToken,
            ':expires_at' => $expiresAt,
            ':id' => $instance['id'],
        ]);

        return $accessToken;
    }

    private function getReportConfig(int $instanceId, string $key): ?array
    {
        $stmt = $this->db->prepare(
            "SELECT report_id, campo_campania, campo_lead_id, campo_email, campo_subject, campo_clicks, campo_date_sent, campo_date_read
             FROM mautic_reportes_map
             WHERE mautic_instancia_id = :instancia_id AND report_key = :report_key AND activo = 1
             LIMIT 1"
        );
        $stmt->execute([
            ':instancia_id' => $instanceId,
            ':report_key' => $key,
        ]);
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        return $row ?: null;
    }

    private function syncReportEvents(
        MauticApiClient $client,
        int $instanceId,
        array $reportConfig,
        ?string $lastReportSyncAt,
        ?int $onlyCampaignId,
        ?string $windowEnd,
        bool $skipActiveFilter,
        ?string $forceReportSyncAt
    ): int
    {
        $reportId = (int)$reportConfig['report_id'];
        $rows = $this->getReportData($client, $reportId);
        if (empty($rows)) {
            return 0;
        }

        $campanias = $this->fetchCampaniasMap($instanceId);
        $events = 0;
        $totals = [];
        $maxSentDate = null;
        $lastReportSyncAt = $this->parseReportDate($lastReportSyncAt);
        $windowEnd = $this->parseReportDate($windowEnd);

        foreach ($rows as $row) {
            $mauticCampaignId = $this->getRowValue($row, $reportConfig['campo_campania']);
            $sentDate = $this->parseReportDate($this->getRowValue($row, $reportConfig['campo_date_sent']));
            $contactoEmailId = null;
            $leadField = $reportConfig['campo_lead_id'] ?? null;
            $leadId = null;
            if ($leadField) {
                $leadId = $this->getRowValue($row, $leadField);
            }
            $contactoId = null;
            $email = $this->getRowValue($row, $reportConfig['campo_email']);
            if ($email) {
                $contactoEmailId = $this->findContactoEmailId($email);
                if ($contactoEmailId) {
                    $contactoId = $this->findContactoIdByEmailId($contactoEmailId);
                }
            }

            if (!$mauticCampaignId) {
                continue;
            }
            if ($onlyCampaignId !== null && (int)$mauticCampaignId !== $onlyCampaignId) {
                continue;
            }
            $campaniaId = $campanias[(int)$mauticCampaignId]['campania_id'] ?? null;
            if (!$campaniaId) {
                continue;
            }
            if (!$skipActiveFilter && !$this->isCampaignActiveNow($campanias[(int)$mauticCampaignId])) {
                continue;
            }

            $subject = null;
            if (!empty($reportConfig['campo_subject'])) {
                $subject = $this->getRowValue($row, $reportConfig['campo_subject']);
                $subject = $subject !== null ? (string)$subject : null;
            }
            if (!$subject) {
                $subject = $this->getRowValue($row, 'subject1');
                $subject = $subject !== null ? (string)$subject : null;
            }
            if (!$subject) {
                $subject = $this->getRowValue($row, 'subject');
                $subject = $subject !== null ? (string)$subject : null;
            }
            if ($subject) {
                $this->updateCampaignSubjectIfEmpty($campaniaId, $subject);
            }

            if ($sentDate && $lastReportSyncAt && $this->compareDate($sentDate, $lastReportSyncAt) <= 0) {
                continue;
            }
            if ($sentDate && $windowEnd && $this->compareDate($sentDate, $windowEnd) > 0) {
                continue;
            }

            if (!$this->isDateInWindow($campanias[(int)$mauticCampaignId], $sentDate)) {
                continue;
            }

            if (!$contactoId) {
                if ($email) {
                    $this->recordPendingEmail(
                        $instanceId,
                        $email,
                        (int)$mauticCampaignId,
                        (int)$campaniaId,
                        $leadId ? (int)$leadId : null,
                        'report'
                    );
                }
                continue;
            }
            if ($leadId && $contactoEmailId) {
                $this->ensureContactMap($instanceId, (int)$contactoId, (int)$leadId);
                $this->ensureContactoEmailMap($instanceId, (int)$contactoEmailId, (int)$leadId);
            }

            if ($sentDate) {
                if ($maxSentDate === null || $this->compareDate($sentDate, $maxSentDate) > 0) {
                    $maxSentDate = $sentDate;
                }
            }

            $clicks = (int)$this->getRowValue($row, $reportConfig['campo_clicks']);
            $readDate = $this->parseReportDate($this->getRowValue($row, $reportConfig['campo_date_read']));
            $opened = $readDate !== null;
            $lastEvent = $readDate ?? $sentDate ?? date('Y-m-d H:i:s');

            $this->assignContactToCampaign($campaniaId, $contactoId);

            $key = $campaniaId . ':' . $contactoId;
            if (!isset($totals[$key])) {
                $totals[$key] = [
                    'campania_id' => $campaniaId,
                    'contacto_id' => $contactoId,
                    'opens' => 'No',
                    'clicks' => 0,
                    'last_event_at' => $lastEvent,
                    'email_subject' => $subject,
                    'contacto_email_id' => $contactoEmailId,
                ];
            }

            if ($opened) {
                $totals[$key]['opens'] = 'Yes';
            }
            if ($clicks > 0) {
                $totals[$key]['clicks'] += $clicks;
                $totals[$key]['opens'] = 'Yes';
            }

            if ($this->compareDate($lastEvent, $totals[$key]['last_event_at']) > 0) {
                $totals[$key]['last_event_at'] = $lastEvent;
            }
            if ($subject) {
                $totals[$key]['email_subject'] = $subject;
            }
            if ($contactoEmailId) {
                $totals[$key]['contacto_email_id'] = $contactoEmailId;
            }
        }

        foreach ($totals as $row) {
            $this->upsertAggregateEvent(
                $instanceId,
                $row['campania_id'],
                $row['contacto_id'],
                $row['opens'],
                $row['clicks'],
                $row['last_event_at'],
                $row['email_subject'],
                $row['contacto_email_id']
            );
            $events++;
        }

        if ($forceReportSyncAt) {
            $this->updateInstanceReportSync($instanceId, $forceReportSyncAt);
            return $events;
        }
        if ($maxSentDate) {
            $this->updateInstanceReportSync($instanceId, $maxSentDate);
        }

        return $events;
    }

    private function getReportData(MauticApiClient $client, int $reportId): array
    {
        $start = 0;
        $limit = 200;
        $items = [];

        do {
            $response = null;
            try {
                $response = $client->get('reports/' . $reportId . '/data', ['start' => $start, 'limit' => $limit]);
            } catch (Throwable $e) {
                try {
                    $response = $client->get('reports/' . $reportId . '/results', ['start' => $start, 'limit' => $limit]);
                } catch (Throwable $e2) {
                    $response = $client->get('reports/' . $reportId, ['start' => $start, 'limit' => $limit]);
                }
            }
            $chunk = $response['data'] ?? $response['results'] ?? $response['rows'] ?? [];
            if (!is_array($chunk)) {
                $chunk = [];
            }
            $items = array_merge($items, $chunk);
            $total = (int)($response['total'] ?? $response['totalResults'] ?? count($chunk));
            $start += $limit;
        } while ($start < $total);

        return $items;
    }

    private function fetchCampaniasMap(int $instanceId): array
    {
        $stmt = $this->db->prepare(
            "SELECT m.mautic_campaign_id, m.campania_id, c.fecha_inicio, c.fecha_fin
             FROM mautic_campania_map m
             JOIN campanias c ON c.id_campania = m.campania_id
             WHERE m.mautic_instancia_id = :instancia_id"
        );
        $stmt->execute([':instancia_id' => $instanceId]);
        $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);
        $out = [];
        foreach ($rows as $row) {
            $out[(int)$row['mautic_campaign_id']] = $row;
        }
        return $out;
    }

    private function getActiveCampaignsFromMap(array $campanias): array
    {
        $active = [];
        foreach ($campanias as $row) {
            if ($this->isCampaignActiveNow($row)) {
                $active[] = $row;
            }
        }
        usort($active, function (array $a, array $b): int {
            $aStart = $a['fecha_inicio'] ?? '';
            $bStart = $b['fecha_inicio'] ?? '';
            if ($aStart !== $bStart) {
                return $aStart <=> $bStart;
            }
            return ((int)$a['mautic_campaign_id']) <=> ((int)$b['mautic_campaign_id']);
        });
        return $active;
    }

    private function getHistoricalCampaignsFromMap(array $campanias): array
    {
        $historical = [];
        $now = date('Y-m-d H:i:s');
        foreach ($campanias as $row) {
            if (empty($row['fecha_fin'])) {
                continue;
            }
            $end = $row['fecha_fin'] . ' 23:59:59';
            if ($end < $now) {
                $historical[] = $row;
            }
        }
        usort($historical, function (array $a, array $b): int {
            $aStart = $a['fecha_inicio'] ?? '';
            $bStart = $b['fecha_inicio'] ?? '';
            if ($aStart !== $bStart) {
                return $aStart <=> $bStart;
            }
            return ((int)$a['mautic_campaign_id']) <=> ((int)$b['mautic_campaign_id']);
        });
        return $historical;
    }

    private function hasCampaignEvents(int $instanceId, int $campaniaId): bool
    {
        $stmt = $this->db->prepare(
            "SELECT 1
             FROM campania_contacto_eventos
             WHERE mautic_instancia_id = :instancia_id
               AND campania_id = :campania_id
             LIMIT 1"
        );
        $stmt->execute([
            ':instancia_id' => $instanceId,
            ':campania_id' => $campaniaId,
        ]);
        return (bool)$stmt->fetchColumn();
    }

    private function getMinCampaignStart(array $campanias): ?string
    {
        $min = null;
        foreach ($campanias as $row) {
            if (empty($row['fecha_inicio'])) {
                continue;
            }
            $value = $row['fecha_inicio'];
            if ($min === null || $value < $min) {
                $min = $value;
            }
        }
        return $min;
    }

    private function offsetDate(string $dateTime, int $seconds): string
    {
        return date('Y-m-d H:i:s', strtotime($dateTime) + $seconds);
    }

    private function getReportDefinition(MauticApiClient $client, int $reportId): array
    {
        return $client->get('reports/' . $reportId);
    }

    private function resolveReportColumn(array $reportDef, string $field): string
    {
        $dataColumns = $reportDef['dataColumns'] ?? [];
        if (isset($dataColumns[$field])) {
            return (string)$dataColumns[$field];
        }
        return $field;
    }

    private function applyReportCampaignFilters(
        MauticApiClient $client,
        array $reportDef,
        array $reportConfig,
        array $campaignRow
    ): void {
        $report = $reportDef['report'] ?? [];
        $filters = $report['filters'] ?? [];
        if (!is_array($filters)) {
            $filters = [];
        }

        $campaignColumn = $this->resolveReportColumn($reportDef, (string)$reportConfig['campo_campania']);
        $dateSentColumn = $this->resolveReportColumn($reportDef, (string)$reportConfig['campo_date_sent']);

        $filters = array_values(array_filter($filters, function ($filter) use ($campaignColumn, $dateSentColumn) {
            if (!is_array($filter)) {
                return true;
            }
            $col = (string)($filter['column'] ?? '');
            return $col !== $campaignColumn && $col !== $dateSentColumn;
        }));

        $filters[] = [
            'column' => $campaignColumn,
            'condition' => 'eq',
            'value' => (string)$campaignRow['mautic_campaign_id'],
            'glue' => 'and',
        ];

        if (!empty($campaignRow['fecha_inicio'])) {
            $filters[] = [
                'column' => $dateSentColumn,
                'condition' => 'gte',
                'value' => $campaignRow['fecha_inicio'] . ' 00:00:00',
                'glue' => 'and',
            ];
        }
        if (!empty($campaignRow['fecha_fin'])) {
            $filters[] = [
                'column' => $dateSentColumn,
                'condition' => 'lte',
                'value' => $campaignRow['fecha_fin'] . ' 23:59:59',
                'glue' => 'and',
            ];
        }

        $payload = [
            'name' => $report['name'] ?? 'All Emails',
            'source' => $report['source'] ?? 'email.stats',
            'columns' => $report['columns'] ?? [],
            'filters' => $filters,
            'tableOrder' => $report['tableOrder'] ?? [],
            'groupBy' => $report['groupBy'] ?? [],
            'settings' => $report['settings'] ?? [],
            'aggregators' => $report['aggregators'] ?? [],
            'graphs' => $report['graphs'] ?? [],
            'isPublished' => $report['isPublished'] ?? true,
            'isScheduled' => $report['isScheduled'] ?? false,
        ];

        $reportId = (int)($report['id'] ?? $reportDef['id'] ?? 0);
        if ($reportId <= 0) {
            return;
        }

        $client->patch('reports/' . $reportId . '/edit', $payload);
    }

    private function applyReportDateFilters(
        MauticApiClient $client,
        array $reportDef,
        array $reportConfig,
        string $start,
        string $end
    ): void {
        $report = $reportDef['report'] ?? [];
        $filters = $report['filters'] ?? [];
        if (!is_array($filters)) {
            $filters = [];
        }

        $dateSentColumn = $this->resolveReportColumn($reportDef, (string)$reportConfig['campo_date_sent']);

        $filters = array_values(array_filter($filters, function ($filter) use ($dateSentColumn) {
            if (!is_array($filter)) {
                return true;
            }
            $col = (string)($filter['column'] ?? '');
            return $col !== $dateSentColumn;
        }));

        $filters[] = [
            'column' => $dateSentColumn,
            'condition' => 'gte',
            'value' => $start,
            'glue' => 'and',
        ];
        $filters[] = [
            'column' => $dateSentColumn,
            'condition' => 'lte',
            'value' => $end,
            'glue' => 'and',
        ];

        $payload = [
            'name' => $report['name'] ?? 'All Emails',
            'source' => $report['source'] ?? 'email.stats',
            'columns' => $report['columns'] ?? [],
            'filters' => $filters,
            'tableOrder' => $report['tableOrder'] ?? [],
            'groupBy' => $report['groupBy'] ?? [],
            'settings' => $report['settings'] ?? [],
            'aggregators' => $report['aggregators'] ?? [],
            'graphs' => $report['graphs'] ?? [],
            'isPublished' => $report['isPublished'] ?? true,
            'isScheduled' => $report['isScheduled'] ?? false,
        ];

        $reportId = (int)($report['id'] ?? $reportDef['id'] ?? 0);
        if ($reportId <= 0) {
            return;
        }

        $client->patch('reports/' . $reportId . '/edit', $payload);
    }

    private function getRowValue(array $row, string $field)
    {
        $field = trim($field);
        if ($field === '') {
            return null;
        }
        if (array_key_exists($field, $row)) {
            return $row[$field];
        }
        $alt = strtolower($field);
        foreach ($row as $key => $value) {
            if (strtolower((string)$key) === $alt) {
                return $value;
            }
        }
        return null;
    }

    private function parseReportDate($value): ?string
    {
        if (!$value) {
            return null;
        }
        $string = (string)$value;
        if (strpos($string, 'T') !== false) {
            $string = str_replace('T', ' ', $string);
        }
        return substr($string, 0, 19);
    }

    private function isDateInWindow(array $campaignRow, ?string $date): bool
    {
        if (!$date) {
            return true;
        }
        $start = !empty($campaignRow['fecha_inicio']) ? $campaignRow['fecha_inicio'] . ' 00:00:00' : null;
        $end = !empty($campaignRow['fecha_fin']) ? $campaignRow['fecha_fin'] . ' 23:59:59' : null;

        if ($start && $date < $start) {
            return false;
        }
        if ($end && $date > $end) {
            return false;
        }
        return true;
    }

    private function isCampaignActiveNow(array $campaignRow): bool
    {
        $now = date('Y-m-d H:i:s');
        $start = !empty($campaignRow['fecha_inicio']) ? $campaignRow['fecha_inicio'] . ' 00:00:00' : null;
        $end = !empty($campaignRow['fecha_fin']) ? $campaignRow['fecha_fin'] . ' 23:59:59' : null;

        if ($start && $now < $start) {
            return false;
        }
        if ($end && $now > $end) {
            return false;
        }
        return true;
    }

    private function compareDate(string $a, string $b): int
    {
        return strcmp($a, $b);
    }

    private function upsertAggregateEvent(
        int $instanceId,
        int $campaniaId,
        int $contactoId,
        string $opens,
        int $clicks,
        string $lastEventAt,
        ?string $emailSubject,
        ?int $contactoEmailId
    ): void {
        $stmt = $this->db->prepare(
            "SELECT id
             FROM campania_contacto_eventos
             WHERE mautic_instancia_id = :instancia_id
               AND campania_id = :campania_id
               AND contacto_id = :contacto_id
             LIMIT 1"
        );
        $stmt->execute([
            ':instancia_id' => $instanceId,
            ':campania_id' => $campaniaId,
            ':contacto_id' => $contactoId,
        ]);
        $row = $stmt->fetch(PDO::FETCH_ASSOC);

        if ($row) {
            $update = $this->db->prepare(
                "UPDATE campania_contacto_eventos
                 SET open = :open,
                     clicks = :clicks,
                     email_subject = :email_subject,
                     contacto_email_id = COALESCE(:contacto_email_id, contacto_email_id),
                     last_event_at = :last_event_at
                 WHERE id = :id"
            );
            $update->execute([
                ':open' => $opens,
                ':clicks' => $clicks,
                ':email_subject' => $emailSubject,
                ':contacto_email_id' => $contactoEmailId,
                ':last_event_at' => $lastEventAt,
                ':id' => $row['id'],
            ]);
            return;
        }

        $insert = $this->db->prepare(
            "INSERT INTO campania_contacto_eventos
                (campania_id, contacto_id, mautic_instancia_id, contacto_email_id, open, clicks, email_subject, last_event_at, created_at)
             VALUES
                (:campania_id, :contacto_id, :instancia_id, :contacto_email_id, :open, :clicks, :email_subject, :last_event_at, NOW())"
        );
        $insert->execute([
            ':campania_id' => $campaniaId,
            ':contacto_id' => $contactoId,
            ':instancia_id' => $instanceId,
            ':contacto_email_id' => $contactoEmailId,
            ':open' => $opens,
            ':clicks' => $clicks,
            ':email_subject' => $emailSubject,
            ':last_event_at' => $lastEventAt,
        ]);
    }

    private function findContactoEmailId(string $email): ?int
    {
        $stmt = $this->db->prepare(
            "SELECT id
             FROM contacto_emails
             WHERE LOWER(email) = :email
             ORDER BY es_principal DESC, estado DESC, id ASC
             LIMIT 1"
        );
        $stmt->execute([':email' => strtolower(trim($email))]);
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        return $row ? (int)$row['id'] : null;
    }

    private function findContactoIdByEmailId(int $contactoEmailId): ?int
    {
        $stmt = $this->db->prepare(
            "SELECT contacto_id
             FROM contacto_emails
             WHERE id = :id
             LIMIT 1"
        );
        $stmt->execute([':id' => $contactoEmailId]);
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        return $row ? (int)$row['contacto_id'] : null;
    }

    private function isCampaignActive(array $campaign): bool
    {
        if (isset($campaign['isPublished']) && (int)$campaign['isPublished'] === 0) {
            return false;
        }
        $now = new DateTime();
        $start = !empty($campaign['publishUp']) ? new DateTime($campaign['publishUp']) : null;
        $end = !empty($campaign['publishDown']) ? new DateTime($campaign['publishDown']) : null;

        if ($start && $now < $start) {
            return false;
        }
        if ($end && $now > $end) {
            return false;
        }
        return true;
    }

    private function ensureContactMap(int $instanceId, int $contactId, int $leadId): void
    {
        $stmt = $this->db->prepare(
            "INSERT IGNORE INTO mautic_contacto_map (mautic_instancia_id, contacto_id, mautic_lead_id, last_sync_at)
             VALUES (:instancia_id, :contacto_id, :lead_id, NOW())"
        );
        $stmt->execute([
            ':instancia_id' => $instanceId,
            ':contacto_id' => $contactId,
            ':lead_id' => $leadId,
        ]);
    }

    private function ensureContactoEmailMap(int $instanceId, int $contactoEmailId, int $leadId): void
    {
        $stmt = $this->db->prepare(
            "INSERT IGNORE INTO mautic_contacto_email_map
                (mautic_instancia_id, contacto_email_id, mautic_lead_id, last_sync_at)
             VALUES (:instancia_id, :contacto_email_id, :lead_id, NOW())"
        );
        $stmt->execute([
            ':instancia_id' => $instanceId,
            ':contacto_email_id' => $contactoEmailId,
            ':lead_id' => $leadId,
        ]);
    }

    private function recordPendingEmail(
        int $instanceId,
        string $email,
        ?int $mauticCampaignId,
        ?int $campaniaId,
        ?int $leadId,
        string $source
    ): void {
        $stmt = $this->db->prepare(
            "INSERT INTO mautic_email_pendiente
                (mautic_instancia_id, email, mautic_campaign_id, campania_id, mautic_lead_id, source, first_seen_at, last_seen_at)
             VALUES
                (:instancia_id, :email, :mautic_campaign_id, :campania_id, :mautic_lead_id, :source, NOW(), NOW())
             ON DUPLICATE KEY UPDATE
                last_seen_at = NOW(),
                campania_id = COALESCE(VALUES(campania_id), campania_id),
                mautic_lead_id = COALESCE(VALUES(mautic_lead_id), mautic_lead_id),
                source = VALUES(source)"
        );
        $stmt->execute([
            ':instancia_id' => $instanceId,
            ':email' => strtolower(trim($email)),
            ':mautic_campaign_id' => $mauticCampaignId,
            ':campania_id' => $campaniaId,
            ':mautic_lead_id' => $leadId,
            ':source' => $source,
        ]);
    }

    private function assignContactToCampaign(int $campaniaId, int $contactoId): void
    {
        $stmt = $this->db->prepare(
            "INSERT IGNORE INTO campania_contacto (campania_id, contacto_id, created_at)
             VALUES (:campania_id, :contacto_id, NOW())"
        );
        $stmt->execute([
            ':campania_id' => $campaniaId,
            ':contacto_id' => $contactoId,
        ]);
    }

    private function updateCampaignSubjectIfEmpty(int $campaniaId, string $subject): void
    {
        $stmt = $this->db->prepare(
            "UPDATE campanias
             SET asunto = :asunto
             WHERE id_campania = :campania_id
               AND (asunto IS NULL OR asunto = '')"
        );
        $stmt->execute([
            ':asunto' => $subject,
            ':campania_id' => $campaniaId,
        ]);
    }
}
