<?php

class RequerimientoV2Model
{
    private PDO $conn;
    private string $table = 'requerimiento';

    public function __construct(PDO $db)
    {
        $this->conn = $db;
    }

    public function list(array $filters = [], int $limit = 25, int $offset = 0): array
    {
        $sql = "SELECT r.*,
                       e.razon_social   AS empresa_razon_social,
                       e.ruc            AS empresa_ruc,
                       e.nombre_comercial AS empresa_nombre_comercial,
                       COALESCE(e.nombre_comercial, e.razon_social) AS empresa_nombre,
                       e.tipo_empresa   AS empresa_tipo_empresa,
                       c.nombres        AS contacto_nombres,
                       c.apellidos      AS contacto_apellidos,
                       CONCAT_WS(' ', c.nombres, c.apellidos) AS contacto_nombre,
                       c.cargo          AS contacto_cargo,
                       ct_main.telefono AS contacto_telefono,
                       ce_main.email    AS contacto_email,
                       l.ruc            AS lead_ruc,
                       l.razon_social   AS lead_razon_social,
                       u.nombres AS usuario_nombres,
                       u.apellidos AS usuario_apellidos,
                       uap.nombres AS usuario_apoyo_nombres,
                       uap.apellidos AS usuario_apoyo_apellidos,
                       h.created_at AS historia_updated_at
                FROM {$this->table} r
                LEFT JOIN empresa e   ON e.id = r.empresa_id
                LEFT JOIN contacto c  ON c.id = r.contacto_id
                LEFT JOIN contacto_telefonos ct_main ON ct_main.contacto_id = c.id AND ct_main.estado = 1 AND ct_main.es_principal = 1
                LEFT JOIN contacto_emails ce_main ON ce_main.contacto_id = c.id AND ce_main.estado = 1 AND ce_main.es_principal = 1
                LEFT JOIN lead l      ON l.id = r.lead_id
                LEFT JOIN (
                    SELECT har.requerimiento_id, har.usuario_id, har.created_at
                    FROM historia_actividad_requerimiento har
                    JOIN (
                        SELECT requerimiento_id, MAX(created_at) AS max_created
                        FROM historia_actividad_requerimiento
                        GROUP BY requerimiento_id
                    ) hmax ON hmax.requerimiento_id = har.requerimiento_id AND hmax.max_created = har.created_at
                ) h ON h.requerimiento_id = r.id
                LEFT JOIN usuarios u ON u.idusuario = h.usuario_id
                LEFT JOIN usuarios uap ON uap.idusuario = r.usuario_apoyo_id
                WHERE (r.estado IS NULL OR r.estado <> '0')
                  AND (r.activo_crm IS NULL OR r.activo_crm = 0)";

        $params = [];
        $q = trim((string)($filters['q'] ?? ''));
        if ($q !== '') {
            $sql .= " AND (r.titulo LIKE :q OR r.descripcion LIKE :q)";
            $params[':q'] = '%' . $q . '%';
        }

        if (!empty($filters['empresa_id'])) {
            $sql .= " AND r.empresa_id = :empresa_id";
            $params[':empresa_id'] = (int)$filters['empresa_id'];
        }

        if (!empty($filters['lead_id'])) {
            $sql .= " AND r.lead_id = :lead_id";
            $params[':lead_id'] = (int)$filters['lead_id'];
        }

        if (!empty($filters['etapa'])) {
            $sql .= " AND r.etapa = :etapa";
            $params[':etapa'] = $filters['etapa'];
        }

        if (isset($filters['estado']) && $filters['estado'] !== '') {
            $sql .= " AND r.estado = :estado";
            $params[':estado'] = $filters['estado'];
        }

        $sql .= " ORDER BY r.created_at DESC, r.id DESC LIMIT :limit OFFSET :offset";
        $stmt = $this->conn->prepare($sql);
        foreach ($params as $k => $v) {
            $stmt->bindValue($k, $v, is_int($v) ? PDO::PARAM_INT : PDO::PARAM_STR);
        }
        $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
        $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
        $stmt->execute();

        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function count(array $filters = []): int
    {
        $sql = "SELECT COUNT(*) FROM {$this->table} r WHERE (r.estado IS NULL OR r.estado <> '0') AND (r.activo_crm IS NULL OR r.activo_crm = 0)";
        $params = [];

        $q = trim((string)($filters['q'] ?? ''));
        if ($q !== '') {
            $sql .= " AND (r.titulo LIKE :q OR r.descripcion LIKE :q)";
            $params[':q'] = '%' . $q . '%';
        }

        if (!empty($filters['empresa_id'])) {
            $sql .= " AND r.empresa_id = :empresa_id";
            $params[':empresa_id'] = (int)$filters['empresa_id'];
        }

        if (!empty($filters['lead_id'])) {
            $sql .= " AND r.lead_id = :lead_id";
            $params[':lead_id'] = (int)$filters['lead_id'];
        }

        if (!empty($filters['etapa'])) {
            $sql .= " AND r.etapa = :etapa";
            $params[':etapa'] = $filters['etapa'];
        }

        if (isset($filters['estado']) && $filters['estado'] !== '') {
            $sql .= " AND r.estado = :estado";
            $params[':estado'] = $filters['estado'];
        }

        $stmt = $this->conn->prepare($sql);
        foreach ($params as $k => $v) {
            $stmt->bindValue($k, $v, is_int($v) ? PDO::PARAM_INT : PDO::PARAM_STR);
        }
        $stmt->execute();
        return (int)$stmt->fetchColumn();
    }

    public function get(int $id): ?array
    {
        $sql = "SELECT r.*,
                       e.razon_social   AS empresa_razon_social,
                       e.ruc            AS empresa_ruc,
                       e.nombre_comercial AS empresa_nombre_comercial,
                       COALESCE(e.nombre_comercial, e.razon_social) AS empresa_nombre,
                       e.tipo_empresa   AS empresa_tipo_empresa,
                       c.nombres        AS contacto_nombres,
                       c.apellidos      AS contacto_apellidos,
                       CONCAT_WS(' ', c.nombres, c.apellidos) AS contacto_nombre,
                       c.cargo          AS contacto_cargo,
                       ct_main.telefono AS contacto_telefono,
                       ce_main.email    AS contacto_email,
                       l.ruc            AS lead_ruc,
                       l.razon_social   AS lead_razon_social,
                       u.nombres AS usuario_nombres,
                       u.apellidos AS usuario_apellidos,
                       uap.nombres AS usuario_apoyo_nombres,
                       uap.apellidos AS usuario_apoyo_apellidos,
                       h.created_at AS historia_updated_at
                FROM {$this->table} r
                LEFT JOIN empresa e   ON e.id = r.empresa_id
                LEFT JOIN contacto c  ON c.id = r.contacto_id
                LEFT JOIN contacto_telefonos ct_main ON ct_main.contacto_id = c.id AND ct_main.estado = 1 AND ct_main.es_principal = 1
                LEFT JOIN contacto_emails ce_main ON ce_main.contacto_id = c.id AND ce_main.estado = 1 AND ce_main.es_principal = 1
                LEFT JOIN lead l      ON l.id = r.lead_id
                LEFT JOIN (
                    SELECT har.requerimiento_id, har.usuario_id, har.created_at
                    FROM historia_actividad_requerimiento har
                    JOIN (
                        SELECT requerimiento_id, MAX(created_at) AS max_created
                        FROM historia_actividad_requerimiento
                        GROUP BY requerimiento_id
                    ) hmax ON hmax.requerimiento_id = har.requerimiento_id AND hmax.max_created = har.created_at
                ) h ON h.requerimiento_id = r.id
                LEFT JOIN usuarios u ON u.idusuario = h.usuario_id
                LEFT JOIN usuarios uap ON uap.idusuario = r.usuario_apoyo_id
                WHERE r.id = :id
                LIMIT 1";
        $stmt = $this->conn->prepare($sql);
        $stmt->bindValue(':id', $id, PDO::PARAM_INT);
        $stmt->execute();
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        return $row ?: null;
    }

    public function existsByLead(int $leadId, ?int $excludeId = null): ?int
    {
        $sql = "SELECT id FROM {$this->table} WHERE lead_id = :lead_id";
        $params = [':lead_id' => $leadId];
        if ($excludeId) {
            $sql .= " AND id <> :exclude_id";
            $params[':exclude_id'] = $excludeId;
        }
        $sql .= " LIMIT 1";
        $stmt = $this->conn->prepare($sql);
        foreach ($params as $k => $v) {
            $stmt->bindValue($k, $v, PDO::PARAM_INT);
        }
        $stmt->execute();
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        return $row ? (int)$row['id'] : null;
    }

    public function create(array $data): int
    {
        $empresaId = (int)($data['empresa_id'] ?? 0);
        $titulo = trim((string)($data['titulo'] ?? ''));
        if ($empresaId <= 0 || $titulo === '') {
            throw new InvalidArgumentException('empresa_id y titulo son obligatorios');
        }

        $leadId = isset($data['lead_id']) && $data['lead_id'] !== '' ? (int)$data['lead_id'] : null;
        if ($leadId) {
            $found = $this->existsByLead($leadId);
            if ($found) {
                throw new RuntimeException("El lead ya tiene un requerimiento (ID {$found})");
            }
        }

        $stmt = $this->conn->prepare(
            "INSERT INTO {$this->table} 
                (empresa_id, lead_id, contacto_id, titulo, descripcion, monto_estimado, monto_real,
                 tipo_condicion_proyecto, motivo_compra, forma_pago, nivel_tentativa, categoria_productos, usuario_apoyo_id,
                 cotizacion_aprobada, escalado_proyecto, activo_crm,
                 fecha_requerida, etapa, origen, prioridad, estado, created_at, updated_at)
             VALUES 
                (:empresa_id, :lead_id, :contacto_id, :titulo, :descripcion, :monto_estimado, :monto_real,
                 :tipo_condicion_proyecto, :motivo_compra, :forma_pago, :nivel_tentativa, :categoria_productos, :usuario_apoyo_id,
                 :cotizacion_aprobada, :escalado_proyecto, :activo_crm,
                 :fecha_requerida, :etapa, :origen, :prioridad, :estado, :created_at, :updated_at)"
        );

        $now = date('Y-m-d H:i:s');
        $stmt->execute([
            ':empresa_id'      => $empresaId,
            ':lead_id'         => $leadId,
            ':contacto_id'     => $data['contacto_id'] ?? null,
            ':titulo'          => $titulo,
            ':descripcion'     => $data['descripcion'] ?? null,
            ':monto_estimado'  => $data['monto_estimado'] ?? null,
            ':monto_real'      => $data['monto_real'] ?? null,
            ':tipo_condicion_proyecto' => $data['tipo_condicion_proyecto'] ?? 'Estandar',
            ':motivo_compra'   => $data['motivo_compra'] ?? null,
            ':forma_pago'      => $data['forma_pago'] ?? null,
            ':nivel_tentativa' => $data['nivel_tentativa'] ?? 'Medio',
            ':categoria_productos' => $data['categoria_productos'] ?? null,
            ':usuario_apoyo_id' => $data['usuario_apoyo_id'] ?? null,
            ':cotizacion_aprobada' => !empty($data['cotizacion_aprobada']) ? 1 : 0,
            ':escalado_proyecto' => !empty($data['escalado_proyecto']) ? 1 : 0,
            ':activo_crm'      => !empty($data['activo_crm']) ? 1 : 0,
            ':fecha_requerida' => $data['fecha_requerida'] ?? null,
            ':etapa'           => $data['etapa'] ?? 'Pendiente Atender',
            ':origen'          => $data['origen'] ?? null,
            ':prioridad'       => $data['prioridad'] ?? 'Media',
            ':estado'          => $data['estado'] ?? '1',
            ':created_at'      => $now,
            ':updated_at'      => $now,
        ]);

        return (int)$this->conn->lastInsertId();
    }

    public function update(int $id, array $data): bool
    {
        $fields = [];
        $params = [':id' => $id];

        $allowed = [
            'empresa_id', 'lead_id', 'contacto_id', 'titulo', 'descripcion',
            'monto_estimado', 'monto_real', 'tipo_condicion_proyecto', 'motivo_compra',
            'forma_pago', 'nivel_tentativa', 'categoria_productos', 'usuario_apoyo_id',
            'cotizacion_aprobada', 'escalado_proyecto', 'activo_crm',
            'fecha_requerida', 'etapa', 'origen', 'prioridad', 'estado'
        ];

        foreach ($allowed as $field) {
            if (array_key_exists($field, $data)) {
                $fields[] = " {$field} = :{$field}";
                $params[":{$field}"] = $data[$field];
            }
        }

        if (empty($fields)) {
            return false;
        }

        if (isset($data['lead_id']) && $data['lead_id'] !== null && $data['lead_id'] !== '') {
            $leadId = (int)$data['lead_id'];
            $found = $this->existsByLead($leadId, $id);
            if ($found) {
                throw new RuntimeException("El lead ya tiene un requerimiento (ID {$found})");
            }
        }

        $fields[] = " updated_at = :updated_at";
        $params[':updated_at'] = date('Y-m-d H:i:s');

        $sql = "UPDATE {$this->table} SET" . implode(',', $fields) . " WHERE id = :id";
        $stmt = $this->conn->prepare($sql);
        return $stmt->execute($params);
    }

    public function delete(int $id): bool
    {
        $stmt = $this->conn->prepare(
            "UPDATE {$this->table} 
             SET estado = '0', deleted_at = :deleted_at 
             WHERE id = :id"
        );
        return $stmt->execute([
            ':deleted_at' => date('Y-m-d H:i:s'),
            ':id' => $id,
        ]);
    }
}
