﻿<?php
class ProveedorModel {
    private $conn;
    private $table = 'proveedores';

    public function __construct(PDO $db) {
        $this->conn = $db;
    }

    public function list(array $filters = [], int $limit = 25, int $offset = 0) {
        $sql = "SELECT * FROM {$this->table} WHERE status = 1";
        $params = [];

        if (!empty($filters['ruc'])) {
            $sql .= " AND ruc = :ruc";
            $params[':ruc'] = $filters['ruc'];
        }
        if (!empty($filters['q'])) {
            $sql .= " AND (razon_social LIKE :q OR razon_comercial LIKE :q)";
            $params[':q'] = '%' . $filters['q'] . '%';
        }

        $sql .= " ORDER BY dateupdate DESC, datecreated DESC LIMIT :limit OFFSET :offset";
        $stmt = $this->conn->prepare($sql);
        foreach ($params as $k => $v) {
            $stmt->bindValue($k, $v, is_int($v) ? PDO::PARAM_INT : PDO::PARAM_STR);
        }
        $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
        $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function count(array $filters = []) {
        $sql = "SELECT COUNT(*) AS c FROM {$this->table} WHERE status = 1";
        $params = [];
        if (!empty($filters['ruc'])) {
            $sql .= " AND ruc = :ruc";
            $params[':ruc'] = $filters['ruc'];
        }
        if (!empty($filters['q'])) {
            $sql .= " AND (razon_social LIKE :q OR razon_comercial LIKE :q)";
            $params[':q'] = '%' . $filters['q'] . '%';
        }
        $stmt = $this->conn->prepare($sql);
        foreach ($params as $k => $v) {
            $stmt->bindValue($k, $v, is_int($v) ? PDO::PARAM_INT : PDO::PARAM_STR);
        }
        $stmt->execute();
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        return (int)($row['c'] ?? 0);
    }

    public function get(int $id) {
        $stmt = $this->conn->prepare("SELECT * FROM {$this->table} WHERE idprovee = :id LIMIT 1");
        $stmt->bindValue(':id', $id, PDO::PARAM_INT);
        $stmt->execute();
        return $stmt->fetch(PDO::FETCH_ASSOC) ?: null;
    }

    public function getByRuc(string $ruc) {
        $stmt = $this->conn->prepare("SELECT * FROM {$this->table} WHERE ruc = :ruc LIMIT 1");
        $stmt->bindValue(':ruc', $ruc, PDO::PARAM_STR);
        $stmt->execute();
        return $stmt->fetch(PDO::FETCH_ASSOC) ?: null;
    }

    public function create(array $data) {
        $columns = [
            'ruc','razon_comercial','razon_social','rubro_sector','tipo_provee','calificacion','telefono','celular','email',
            'aniversario','perfil','pais','distrito','ciudad','direccion','Referencia','catalogos','redes_sociales',
            'paginaweb','origen','condicion_contribuyente','estado_contribuyente'
        ];

        $fields = [];
        $placeholders = [];
        $params = [];
        foreach ($columns as $c) {
            if (array_key_exists($c, $data)) {
                $fields[] = $c;
                $placeholders[] = ':' . $c;
                $params[':' . $c] = $data[$c];
            }
        }
        if (empty($fields)) {
            throw new InvalidArgumentException('No hay campos para insertar.');
        }
        
        // IMPORTANTE: user_id es obligatorio y se pasa desde el controlador (de la sesión)
        if (!isset($data['user_id'])) {
            throw new InvalidArgumentException('user_id es requerido para crear un proveedor.');
        }
        $fields[] = 'user_id';
        $placeholders[] = ':user_id';
        $params[':user_id'] = $data['user_id'];
        
        $sql = 'INSERT INTO ' . $this->table . ' (' . implode(',', $fields) . ') VALUES (' . implode(',', $placeholders) . ')';
        $stmt = $this->conn->prepare($sql);
        foreach ($params as $k => $v) {
            $stmt->bindValue($k, $v);
        }
        $stmt->execute();
        return (int)$this->conn->lastInsertId();
    }

    public function update(int $id, array $data) {
        $columns = [
            'ruc','razon_comercial','razon_social','rubro_sector','tipo_provee','calificacion','telefono','celular','email',
            'aniversario','perfil','pais','distrito','ciudad','direccion','Referencia','catalogos','redes_sociales',
            'paginaweb','origen','condicion_contribuyente','estado_contribuyente','user_id'
        ];
        $sets = [];
        $params = [':id' => $id];
        foreach ($columns as $c) {
            if (array_key_exists($c, $data)) {
                $sets[] = $c . ' = :' . $c;
                $params[':' . $c] = $data[$c];
            }
        }
        if (empty($sets)) return false;
        $sql = 'UPDATE ' . $this->table . ' SET ' . implode(',', $sets) . ' WHERE idprovee = :id';
        $stmt = $this->conn->prepare($sql);
        foreach ($params as $k => $v) $stmt->bindValue($k, $v);
        return $stmt->execute();
    }

    public function delete(int $id) {
        // Eliminación lógica: cambiar status de 1 a 0 (proveedor no visible)
        // en lugar de eliminar físicamente el registro
        $stmt = $this->conn->prepare("UPDATE {$this->table} SET status = 0 WHERE idprovee = :id");
        $stmt->bindValue(':id', $id, PDO::PARAM_INT);
        return $stmt->execute();
    }
}
