<?php

class ProductModel
{
    private PDO $conn;
    private string $table = 'product_model';

    public function __construct(PDO $db)
    {
        $this->conn = $db;
    }

    public function list(string $q = ''): array
    {
        $sql = "SELECT pm.*, c.name_cat AS categoria_nombre
                FROM {$this->table} pm
                LEFT JOIN categoria c ON pm.catg_id = c.idcat
                WHERE pm.status = 1";
        $params = [];
        if ($q !== '') {
            $sql .= " AND (pm.name_prod LIKE :q OR pm.variant_name LIKE :q OR pm.variant_code LIKE :q)";
            $params[':q'] = '%' . $q . '%';
        }
        $sql .= " ORDER BY pm.dateupdate DESC, pm.datecreate DESC, pm.idprodmodel DESC";
        $stmt = $this->conn->prepare($sql);
        foreach ($params as $k => $v) {
            $stmt->bindValue($k, $v, PDO::PARAM_STR);
        }
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function get(int $id): ?array
    {
        $stmt = $this->conn->prepare("SELECT * FROM {$this->table} WHERE idprodmodel = :id LIMIT 1");
        $stmt->bindValue(':id', $id, PDO::PARAM_INT);
        $stmt->execute();
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        return $row ?: null;
    }

    public function create(array $data): int
    {
        if (!isset($data['name_prod']) || trim((string)$data['name_prod']) === '') {
            throw new InvalidArgumentException('El nombre del producto es obligatorio.');
        }
        if (!isset($data['catg_id']) || (int)$data['catg_id'] <= 0) {
            throw new InvalidArgumentException('La categoría principal es obligatoria.');
        }

        $columns = [
            'name_prod',
            'catg_id',
            'catg_ids',
            'variant_code',
            'variant_name',
            'detail',
            'img',
            'status',
        ];

        $fields = [];
        $placeholders = [];
        $params = [];

        foreach ($columns as $c) {
            if (array_key_exists($c, $data)) {
                $fields[] = $c;
                $placeholders[] = ':' . $c;
                $params[':' . $c] = $data[$c];
            }
        }

        if (!isset($data['status']) || $data['status'] === '') {
            $fields[] = 'status';
            $placeholders[] = ':status';
            $params[':status'] = 1;
        }

        $sql = 'INSERT INTO ' . $this->table . ' (' . implode(',', $fields) . ') VALUES (' . implode(',', $placeholders) . ')';
        $stmt = $this->conn->prepare($sql);
        foreach ($params as $k => $v) {
            $stmt->bindValue($k, $v);
        }
        $stmt->execute();
        return (int)$this->conn->lastInsertId();
    }

    public function update(int $id, array $data): bool
    {
        $columns = [
            'name_prod',
            'catg_id',
            'catg_ids',
            'variant_code',
            'variant_name',
            'detail',
            'img',
            'status',
        ];
        $sets = [];
        $params = [':id' => $id];

        foreach ($columns as $c) {
            if (array_key_exists($c, $data)) {
                $sets[] = $c . ' = :' . $c;
                $params[':' . $c] = $data[$c];
            }
        }

        if (empty($sets)) {
            return false;
        }

        $sql = 'UPDATE ' . $this->table . ' SET ' . implode(',', $sets) . ' WHERE idprodmodel = :id';
        $stmt = $this->conn->prepare($sql);
        foreach ($params as $k => $v) {
            $stmt->bindValue($k, $v);
        }
        return $stmt->execute();
    }

    public function delete(int $id): bool
    {
        $stmt = $this->conn->prepare("UPDATE {$this->table} SET status = 2 WHERE idprodmodel = :id");
        $stmt->bindValue(':id', $id, PDO::PARAM_INT);
        return $stmt->execute();
    }
}
