<?php
class EmpresaModel
{
    private PDO $conn;
    private string $table = 'empresa';

    public function __construct(PDO $db)
    {
        $this->conn = $db;
    }

    public function list(array $filters = [], int $limit = 25, int $offset = 0): array
    {
        $sql = "SELECT id, ruc, nombre_comercial, razon_social, rubro, telefono, email
                FROM {$this->table}
                WHERE estado = 1";
        $params = [];

        $q = trim((string)($filters['q'] ?? ''));
        $by = trim((string)($filters['by'] ?? 'todos'));
        if ($q !== '') {
            $like = '%' . $q . '%';
            switch ($by) {
                case 'id':
                    if (ctype_digit($q)) {
                        $sql .= " AND id = :id";
                        $params[':id'] = (int)$q;
                    }
                    break;
                case 'ruc':
                    $sql .= " AND ruc LIKE :q";
                    $params[':q'] = $like;
                    break;
                case 'razon_social':
                    $sql .= " AND razon_social LIKE :q";
                    $params[':q'] = $like;
                    break;
                case 'nombre_comercial':
                    $sql .= " AND nombre_comercial LIKE :q";
                    $params[':q'] = $like;
                    break;
                case 'email':
                    $sql .= " AND email LIKE :q";
                    $params[':q'] = $like;
                    break;
                case 'telefono':
                    $sql .= " AND (telefono LIKE :q OR celular LIKE :q)";
                    $params[':q'] = $like;
                    break;
                case 'rubro':
                    $sql .= " AND rubro LIKE :q";
                    $params[':q'] = $like;
                    break;
                case 'todos':
                default:
                    $sql .= " AND (ruc LIKE :q OR razon_social LIKE :q OR nombre_comercial LIKE :q
                               OR rubro LIKE :q OR telefono LIKE :q OR celular LIKE :q OR email LIKE :q)";
                    $params[':q'] = $like;
                    break;
            }
        }

        $sql .= " ORDER BY updated_at DESC, created_at DESC, id DESC LIMIT :limit OFFSET :offset";
        $stmt = $this->conn->prepare($sql);
        foreach ($params as $k => $v) {
            $stmt->bindValue($k, $v, is_int($v) ? PDO::PARAM_INT : PDO::PARAM_STR);
        }
        $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
        $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function count(array $filters = []): int
    {
        $sql = "SELECT COUNT(*) FROM {$this->table} WHERE estado = 1";
        $params = [];

        $q = trim((string)($filters['q'] ?? ''));
        $by = trim((string)($filters['by'] ?? 'todos'));
        if ($q !== '') {
            $like = '%' . $q . '%';
            switch ($by) {
                case 'id':
                    if (ctype_digit($q)) {
                        $sql .= " AND id = :id";
                        $params[':id'] = (int)$q;
                    }
                    break;
                case 'ruc':
                    $sql .= " AND ruc LIKE :q";
                    $params[':q'] = $like;
                    break;
                case 'razon_social':
                    $sql .= " AND razon_social LIKE :q";
                    $params[':q'] = $like;
                    break;
                case 'nombre_comercial':
                    $sql .= " AND nombre_comercial LIKE :q";
                    $params[':q'] = $like;
                    break;
                case 'email':
                    $sql .= " AND email LIKE :q";
                    $params[':q'] = $like;
                    break;
                case 'telefono':
                    $sql .= " AND (telefono LIKE :q OR celular LIKE :q)";
                    $params[':q'] = $like;
                    break;
                case 'rubro':
                    $sql .= " AND rubro LIKE :q";
                    $params[':q'] = $like;
                    break;
                case 'todos':
                default:
                    $sql .= " AND (ruc LIKE :q OR razon_social LIKE :q OR nombre_comercial LIKE :q
                               OR rubro LIKE :q OR telefono LIKE :q OR celular LIKE :q OR email LIKE :q)";
                    $params[':q'] = $like;
                    break;
            }
        }

        $stmt = $this->conn->prepare($sql);
        foreach ($params as $k => $v) {
            $stmt->bindValue($k, $v, is_int($v) ? PDO::PARAM_INT : PDO::PARAM_STR);
        }
        $stmt->execute();
        return (int)$stmt->fetchColumn();
    }

    public function get(int $id): ?array
    {
        $stmt = $this->conn->prepare("SELECT * FROM {$this->table} WHERE id = :id LIMIT 1");
        $stmt->bindValue(':id', $id, PDO::PARAM_INT);
        $stmt->execute();
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        return $row ?: null;
    }

    public function create(array $data): int
    {
        $fields = $this->normalizePayload($data);
        $fields['created_at'] = date('Y-m-d H:i:s');
        $fields['updated_at'] = $fields['created_at'];

        $columns = array_keys($fields);
        $placeholders = array_map(fn($c) => ':' . $c, $columns);
        $sql = "INSERT INTO {$this->table} (" . implode(',', $columns) . ") VALUES (" . implode(',', $placeholders) . ")";
        $stmt = $this->conn->prepare($sql);
        foreach ($fields as $k => $v) {
            $stmt->bindValue(':' . $k, $v);
        }
        $stmt->execute();
        return (int)$this->conn->lastInsertId();
    }

    public function update(int $id, array $data): bool
    {
        $fields = $this->normalizePayload($data);
        $fields['updated_at'] = date('Y-m-d H:i:s');

        $sets = [];
        foreach ($fields as $k => $v) {
            $sets[] = $k . ' = :' . $k;
        }
        $sql = "UPDATE {$this->table} SET " . implode(', ', $sets) . " WHERE id = :id";
        $stmt = $this->conn->prepare($sql);
        $stmt->bindValue(':id', $id, PDO::PARAM_INT);
        foreach ($fields as $k => $v) {
            $stmt->bindValue(':' . $k, $v);
        }
        return $stmt->execute();
    }

    public function delete(int $id): bool
    {
        $sql = "UPDATE {$this->table} SET estado = 0, deleted_at = :deleted_at WHERE id = :id";
        $stmt = $this->conn->prepare($sql);
        $stmt->bindValue(':id', $id, PDO::PARAM_INT);
        $stmt->bindValue(':deleted_at', date('Y-m-d H:i:s'));
        return $stmt->execute();
    }

    private function normalizePayload(array $data): array
    {
        $fields = [
            'ruc' => trim((string)($data['ruc'] ?? '')),
            'nombre_comercial' => trim((string)($data['nombre_comercial'] ?? '')),
            'razon_social' => trim((string)($data['razon_social'] ?? '')),
            'rubro' => trim((string)($data['rubro'] ?? '')),
            'tipo_empresa' => $data['tipo_empresa'] ?? null,
            'n_trabajadores' => isset($data['n_trabajadores']) ? (int)$data['n_trabajadores'] : null,
            'estado_contribuyente' => $data['estado_contribuyente'] ?? null,
            'condicion_contribuyente' => $data['condicion_contribuyente'] ?? null,
            'country_id' => trim((string)($data['country_id'] ?? 'PE')),
            'department_id' => trim((string)($data['department_id'] ?? '')),
            'province_id' => trim((string)($data['province_id'] ?? '')),
            'district_id' => trim((string)($data['district_id'] ?? '')),
            'direccion' => trim((string)($data['direccion'] ?? '')),
            'referencia' => trim((string)($data['referencia'] ?? '')),
            'telefono' => trim((string)($data['telefono'] ?? '')),
            'celular' => trim((string)($data['celular'] ?? '')),
            'email' => trim((string)($data['email'] ?? '')),
            'website' => trim((string)($data['website'] ?? '')),
            'origen' => trim((string)($data['origen'] ?? '')),
            'aniversario' => $this->normalizeDate($data['aniversario'] ?? null),
            'politica_pago' => trim((string)($data['politica_pago'] ?? '')),
            'procedimiento_especial' => trim((string)($data['procedimiento_especial'] ?? '')),
            'frecuencia_compra' => trim((string)($data['frecuencia_compra'] ?? '')),
            'adicionales' => trim((string)($data['adicionales'] ?? '')),
            'trabaja_con_proveedores' => !empty($data['trabaja_con_proveedores']) ? 1 : 0,
        ];

        // Limpiar valores vacios a NULL donde aplica
        foreach (['tipo_empresa','estado_contribuyente','condicion_contribuyente','aniversario'] as $nullable) {
            if ($fields[$nullable] === '') {
                $fields[$nullable] = null;
            }
        }

        return $fields;
    }

    private function normalizeDate($value): ?string
    {
        $value = trim((string)$value);
        if ($value === '') {
            return null;
        }
        $time = strtotime($value);
        if ($time === false) {
            return null;
        }
        return date('Y-m-d', $time);
    }
}
