<?php

class EmailCliente
{
    private $conn;
    private $table_name = "groupemail";
    private $relation;

    public $id;
    public $email;

    public function __construct(PDO $db, $relation = "clientes")
    {
        $this->conn = $db;
        $this->relation = $relation;
    }

    /**
     * Obtiene los antecedentes de un cliente por su ID
     * @param int $id
     * @return array
     */
    public function getByClienteId($id)
    {
        $sql = "SELECT 
            ge.idmail AS id,            
            ge.email AS email
        FROM {$this->table_name} ge
        WHERE ge.idtabla = :idcliente AND ge.tabla = :tabla
        ORDER BY ge.idmail";
        $stmt = $this->conn->prepare($sql);
        $stmt->bindValue(':idcliente', $id, PDO::PARAM_INT);
        $stmt->bindValue(':tabla',   $this->relation, PDO::PARAM_STR);
        $stmt->execute();
        $emails = $stmt->fetchAll(PDO::FETCH_ASSOC);
        // Formatear resultados, si uno de los emails es NULL o si no tiene formato de email se borra
        // foreach ($emails as $key => $email) {
        //     if (empty($email['email']) || !filter_var($email['email'], FILTER_VALIDATE_EMAIL)) {
        //         unset($emails[$key]);
        //     } else {
        //         $emails[$key]['email'] = strtolower(trim($email['email']));
        //     }
        // }
        return $emails;
    }

    /**
     * Inserta un nuevo antecedente si no existe
     * @param int $idcliente — ID del cliente, no nulo
     * @param int $idusuario — ID del usuario, no nulo
     * @param string $emails — Texto del antecedente, no nulo
     * @return array — true si OK, mensaje de error si falla
     */
    public function update($idcliente, $emails, $sync_source = null)
    {
        // Validar parámetros
        if (empty($idcliente) || empty($emails)) {
            return [
                'success' => false,
                'message' => 'ID del cliente y emails son obligatorios'
            ];
        }

        // Validar que el ID del cliente sea un número entero
        if (!is_int($idcliente) || $idcliente <= 0) {
            return [
                'success' => false,
                'message' => 'ID del cliente debe ser un número entero positivo'
            ];
        }

        // Validar que el ID del cliente exista
        $query = "SELECT COUNT(*) FROM {$this->relation} WHERE idcliente = :idcliente";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':idcliente', $idcliente, PDO::PARAM_INT);
        $stmt->execute();
        if ($stmt->fetchColumn() == 0) {
            return [
                'success' => false,
                'message' => 'El ID del cliente no existe'
            ];
        }

        // Convertir el string de emails en un array si no lo es ya
        if (is_string($emails)) {
            $emails = explode(',', $emails);
        } elseif (!is_array($emails)) {
            return [
                'success' => false,
                'message' => 'El formato de los emails no es válido'
            ];
        }

        // Normalizar y limpiar correos
        $emails = array_values(array_filter(array_unique(array_map(function ($email) {
            return strtolower(trim($email));
        }, $emails)), function ($email) {
            return $email !== '';
        }));

        if (empty($emails)) {
            return [
                'success' => false,
                'message' => 'No se recibieron correos válidos'
            ];
        }

        // Validar correos y detectar conflictos con otros clientes antes de modificar datos
        $conflictos = [];
        $checkQuery = "SELECT idtabla FROM {$this->table_name} WHERE tabla = :tabla AND LOWER(email) = :email LIMIT 1";
        $checkStmt = $this->conn->prepare($checkQuery);
        $clienteExisteStmt = $this->conn->prepare("SELECT COUNT(*) FROM {$this->relation} WHERE idcliente = :idcliente");
        $deleteStaleStmt = $this->conn->prepare("DELETE FROM {$this->table_name} WHERE tabla = :tabla AND idtabla = :idcliente AND LOWER(email) = :email");
        foreach ($emails as $email) {
            if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
                return [
                    'success' => false,
                    'message' => "El email '$email' no es válido"
                ];
            }
            $checkStmt->execute([
                ':tabla' => $this->relation,
                ':email' => $email
            ]);
            $existingId = $checkStmt->fetchColumn();
            if ($existingId && (int)$existingId !== (int)$idcliente) {
                // Si el ID que tiene ese email ya no existe en clientes, limpialo y permite registrar
                $clienteExisteStmt->execute([':idcliente' => (int)$existingId]);
                $clienteSigueExistiendo = (int)$clienteExisteStmt->fetchColumn() > 0;

                if ($clienteSigueExistiendo) {
                    $conflictos[] = ['email' => $email, 'idcliente' => (int)$existingId];
                } else {
                    // Registro huérfano: eliminarlo para liberar el email
                    $deleteStaleStmt->execute([
                        ':tabla' => $this->relation,
                        ':idcliente' => (int)$existingId,
                        ':email' => $email
                    ]);
                }
            }
        }

        if (!empty($conflictos)) {
            $lista = implode(', ', array_map(
                fn ($c) => "{$c['email']} (cliente #{$c['idcliente']})",
                $conflictos
            ));
            return [
                'success' => false,
                'message' => "Alerta de duplicidad. No se puede actualizar: Los siguientes correos ya están asociados a otro cliente: {$lista}"
            ];
        }

        
        $sqlDelete =  "DELETE FROM {$this->table_name}  WHERE idtabla = :idcliente AND tabla = :tabla";
        $checkStmtDelete = $this->conn->prepare($sqlDelete);
        $checkStmtDelete->bindParam(':idcliente', $idcliente, PDO::PARAM_INT);
        $checkStmtDelete->bindParam(':tabla', $this->relation, PDO::PARAM_STR);

        // Eliminar emails del cliente
        if (!$checkStmtDelete->execute()) {
            return [
                'success' => false,
                'message' => "No se pudieron eliminar los emails del cliente con ID: '$idcliente'"
            ];
        }
        // Preparar la consulta de inserción
        $sql = "INSERT INTO {$this->table_name} (tabla, idtabla, email) VALUES (:tabla, :idcliente, :email)";

        $stmt = $this->conn->prepare($sql);
        $insertados = 0;
        foreach ($emails as $email) {
            if (!$stmt->execute([
                ':tabla' => $this->relation,
                ':idcliente' => $idcliente,
                ':email' => $email
            ])) {
                return [
                    'success' => false,
                    'message' => "Error al insertar el email '$email'"
                ];
            }
            $insertados++;
        }

        if ($insertados === 0) {
            return [
                'success' => false,
                'message' => 'No se registró ningún correo para el cliente'
            ];
        }

        return [
            'success' => true,
            'message' => 'Emails insertados correctamente',
            'inserted' => $insertados
        ];
    }
}
