<?php
class ContactoTelefonoModel
{
    private PDO $conn;
    private string $table = 'contacto_telefonos';

    public function __construct(PDO $db)
    {
        $this->conn = $db;
    }

    public function listByContacto(int $contactoId): array
    {
        $stmt = $this->conn->prepare(
            "SELECT id, telefono, tipo, es_principal
             FROM {$this->table}
             WHERE contacto_id = :contacto_id AND estado = 1
             ORDER BY es_principal DESC, id ASC"
        );
        $stmt->bindValue(':contacto_id', $contactoId, PDO::PARAM_INT);
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function sync(int $contactoId, array $telefonos): void
    {
        $normalized = $this->normalizeTelefonos($telefonos);
        $existing = $this->fetchExisting($contactoId);
        $now = date('Y-m-d H:i:s');

        $active = [];
        foreach ($existing as $row) {
            if ((int)$row['estado'] === 1) {
                $active[$row['telefono']] = $row;
            }
        }

        $input = [];
        foreach ($normalized as $row) {
            $input[$row['telefono']] = $row;
        }

        foreach ($active as $telefono => $row) {
            if (!isset($input[$telefono])) {
                $this->setEstado($row['id'], 0, $now);
            }
        }

        foreach ($input as $telefono => $row) {
            if (isset($existing[$telefono])) {
                $this->updateRow($existing[$telefono]['id'], $row, $now);
            } else {
                $this->insertRow($contactoId, $row, $now);
            }
        }
    }

    private function normalizeTelefonos(array $telefonos): array
    {
        $allowedTipos = ['movil', 'fijo', 'trabajo', 'casa', 'otro'];
        $out = [];
        foreach ($telefonos as $item) {
            if (is_string($item)) {
                $telefono = $this->normalizePhone($item);
                $tipo = 'otro';
                $principal = 0;
            } else {
                $telefono = $this->normalizePhone((string)($item['telefono'] ?? ''));
                $tipo = strtolower(trim((string)($item['tipo'] ?? 'otro')));
                $principal = !empty($item['es_principal']) ? 1 : 0;
            }
            if ($telefono === '') {
                continue;
            }
            if (!in_array($tipo, $allowedTipos, true)) {
                $tipo = 'otro';
            }
            if (!isset($out[$telefono]) || $principal === 1 || $out[$telefono]['es_principal'] !== 1) {
                $out[$telefono] = ['telefono' => $telefono, 'tipo' => $tipo, 'es_principal' => $principal];
            }
        }
        return array_values($out);
    }

    private function fetchExisting(int $contactoId): array
    {
        $stmt = $this->conn->prepare(
            "SELECT id, telefono, tipo, es_principal, estado
             FROM {$this->table}
             WHERE contacto_id = :contacto_id"
        );
        $stmt->execute([':contacto_id' => $contactoId]);
        $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);
        $byTelefono = [];
        foreach ($rows as $row) {
            $telefono = $row['telefono'];
            if (!isset($byTelefono[$telefono]) || ((int)$byTelefono[$telefono]['estado'] !== 1 && (int)$row['estado'] === 1)) {
                $byTelefono[$telefono] = $row;
            }
        }
        return $byTelefono;
    }

    private function setEstado(int $id, int $estado, string $now): void
    {
        $stmt = $this->conn->prepare(
            "UPDATE {$this->table}
             SET estado = :estado, updated_at = :updated_at, es_principal = 0
             WHERE id = :id"
        );
        $stmt->execute([
            ':estado' => $estado,
            ':updated_at' => $now,
            ':id' => $id,
        ]);
    }

    private function updateRow(int $id, array $row, string $now): void
    {
        $stmt = $this->conn->prepare(
            "UPDATE {$this->table}
             SET tipo = :tipo, es_principal = :es_principal, estado = 1, updated_at = :updated_at
             WHERE id = :id"
        );
        $stmt->execute([
            ':tipo' => $row['tipo'],
            ':es_principal' => $row['es_principal'],
            ':updated_at' => $now,
            ':id' => $id,
        ]);
    }

    private function insertRow(int $contactoId, array $row, string $now): void
    {
        $stmt = $this->conn->prepare(
            "INSERT INTO {$this->table} (contacto_id, telefono, tipo, es_principal, estado, created_at, updated_at)
             VALUES (:contacto_id, :telefono, :tipo, :es_principal, 1, :created_at, :updated_at)"
        );
        $stmt->execute([
            ':contacto_id' => $contactoId,
            ':telefono' => $row['telefono'],
            ':tipo' => $row['tipo'],
            ':es_principal' => $row['es_principal'],
            ':created_at' => $now,
            ':updated_at' => $now,
        ]);
    }

    private function normalizePhone(string $value): string
    {
        return preg_replace('/\D+/', '', $value);
    }
}
