﻿<?php
class ContactoProveedorModel {
    private $conn;
    private $table = 'contacto_provee';

    public function __construct(PDO $db) {
        $this->conn = $db;
    }

    /**
     * Listar contactos por proveedor (sin filtro de estado, para detalle)
     */
    public function listByProveedor(int $proveeId) {
        $stmt = $this->conn->prepare("SELECT * FROM {$this->table} WHERE provee_id = :id ORDER BY idcont_provee");
        $stmt->bindValue(':id', $proveeId, PDO::PARAM_INT);
        $stmt->execute();
        $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);
        foreach ($rows as &$r) {
            $r['telefono'] = $this->decodeJson($r['telefono']);
            $r['celular'] = $this->decodeJson($r['celular']);
            $r['email'] = $this->decodeJson($r['email']);
        }
        return $rows;
    }

    /**
     * Listar contactos con filtros opcionales y paginación (solo estado=1)
     */
    public function list(array $filters = [], int $limit = 25, int $offset = 0) {
        // Traer contactos activos junto con el nombre del proveedor y usuario creador
        $sql = "SELECT cp.*, p.razon_social AS empresa, u.nombres AS user_nombres, u.apellidos AS user_apellidos
                FROM {$this->table} cp
                LEFT JOIN proveedores p ON p.idprovee = cp.provee_id
                LEFT JOIN usuarios u ON u.idusuario = cp.user_id
                WHERE cp.estado = 1";
        $params = [];

        if (!empty($filters['provee_id'])) {
            $sql .= " AND cp.provee_id = :provee_id";
            $params[':provee_id'] = (int)$filters['provee_id'];
        }
        if (!empty($filters['q'])) {
            $q = '%' . $filters['q'] . '%';
            $campo = $filters['campo'] ?? 'todos';
            switch ($campo) {
                case 'id':
                    if (ctype_digit((string)$filters['q'])) {
                        $sql .= " AND cp.idcont_provee = :idexact";
                        $params[':idexact'] = (int)$filters['q'];
                    } else {
                        $sql .= " AND (cp.nombre LIKE :q OR cp.apellido LIKE :q OR cp.cargo LIKE :q)";
                        $params[':q'] = $q;
                    }
                    break;
                case 'nombre':
                    $sql .= " AND cp.nombre LIKE :q"; $params[':q'] = $q; break;
                case 'apellido':
                    $sql .= " AND cp.apellido LIKE :q"; $params[':q'] = $q; break;
                case 'email':
                    $sql .= " AND cp.email LIKE :q"; $params[':q'] = $q; break;
                case 'empresa':
                    $sql .= " AND p.razon_social LIKE :q"; $params[':q'] = $q; break;
                case 'telefono':
                    $sql .= " AND cp.telefono LIKE :q"; $params[':q'] = $q; break;
                case 'celular':
                    $sql .= " AND cp.celular LIKE :q"; $params[':q'] = $q; break;
                case 'usuario':
                    $sql .= " AND ((u.nombres LIKE :q) OR (u.apellidos LIKE :q) OR (CONCAT(u.nombres,' ',u.apellidos) LIKE :q))"; $params[':q'] = $q; break;
                case 'todos':
                default:
                    $sql .= " AND (cp.nombre LIKE :q OR cp.apellido LIKE :q OR cp.cargo LIKE :q OR p.razon_social LIKE :q OR cp.email LIKE :q OR cp.telefono LIKE :q OR cp.celular LIKE :q)";
                    $params[':q'] = $q;
                    break;
            }
        }

        $sql .= " ORDER BY cp.dateupdate DESC, cp.datecreated DESC LIMIT :limit OFFSET :offset";
        $stmt = $this->conn->prepare($sql);
        foreach ($params as $k => $v) {
            $stmt->bindValue($k, $v, is_int($v) ? PDO::PARAM_INT : PDO::PARAM_STR);
        }
        $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
        $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
        $stmt->execute();
        $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);
        foreach ($rows as &$r) {
            $r['telefono'] = $this->decodeJson($r['telefono']);
            $r['celular'] = $this->decodeJson($r['celular']);
            $r['email'] = $this->decodeJson($r['email']);
        }
        return $rows;
    }

    /**
     * Contar contactos con filtros (solo estado=1)
     */
    public function count(array $filters = []) {
        $sql = "SELECT COUNT(*) AS c
                FROM {$this->table} cp
                LEFT JOIN proveedores p ON p.idprovee = cp.provee_id
                LEFT JOIN usuarios u ON u.idusuario = cp.user_id
                WHERE cp.estado = 1";
        $params = [];
        if (!empty($filters['provee_id'])) {
            $sql .= " AND cp.provee_id = :provee_id";
            $params[':provee_id'] = (int)$filters['provee_id'];
        }
        if (!empty($filters['q'])) {
            $q = '%' . $filters['q'] . '%';
            $campo = $filters['campo'] ?? 'todos';
            switch ($campo) {
                case 'id':
                    if (ctype_digit((string)$filters['q'])) {
                        $sql .= " AND cp.idcont_provee = :idexact";
                        $params[':idexact'] = (int)$filters['q'];
                    } else {
                        $sql .= " AND (cp.nombre LIKE :q OR cp.apellido LIKE :q OR cp.cargo LIKE :q)";
                        $params[':q'] = $q;
                    }
                    break;
                case 'nombre':
                    $sql .= " AND cp.nombre LIKE :q"; $params[':q'] = $q; break;
                case 'apellido':
                    $sql .= " AND cp.apellido LIKE :q"; $params[':q'] = $q; break;
                case 'email':
                    $sql .= " AND cp.email LIKE :q"; $params[':q'] = $q; break;
                case 'empresa':
                    $sql .= " AND p.razon_social LIKE :q"; $params[':q'] = $q; break;
                case 'telefono':
                    $sql .= " AND cp.telefono LIKE :q"; $params[':q'] = $q; break;
                case 'celular':
                    $sql .= " AND cp.celular LIKE :q"; $params[':q'] = $q; break;
                case 'usuario':
                    $sql .= " AND ((u.nombres LIKE :q) OR (u.apellidos LIKE :q) OR (CONCAT(u.nombres,' ',u.apellidos) LIKE :q))"; $params[':q'] = $q; break;
                case 'todos':
                default:
                    $sql .= " AND (cp.nombre LIKE :q OR cp.apellido LIKE :q OR cp.cargo LIKE :q OR p.razon_social LIKE :q OR cp.email LIKE :q OR cp.telefono LIKE :q OR cp.celular LIKE :q)";
                    $params[':q'] = $q;
                    break;
            }
        }
        $stmt = $this->conn->prepare($sql);
        foreach ($params as $k => $v) {
            $stmt->bindValue($k, $v, is_int($v) ? PDO::PARAM_INT : PDO::PARAM_STR);
        }
        $stmt->execute();
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        return (int)($row['c'] ?? 0);
    }

    /**
     * Obtener un contacto por ID
     */
    public function get(int $id) {
        $sql = "SELECT cp.*,
                       p.razon_social   AS empresa,
                       p.razon_comercial AS proveedor_razon_comercial,
                       p.ruc            AS proveedor_ruc,
                       u.nombres        AS user_nombres,
                       u.apellidos      AS user_apellidos,
                       TRIM(CONCAT(COALESCE(u.nombres,''), ' ', COALESCE(u.apellidos,''))) AS user_fullname
                FROM {$this->table} cp
                LEFT JOIN proveedores p ON p.idprovee = cp.provee_id
                LEFT JOIN usuarios u ON u.idusuario = cp.user_id
                WHERE cp.idcont_provee = :id
                LIMIT 1";
        $stmt = $this->conn->prepare($sql);
        $stmt->bindValue(':id', $id, PDO::PARAM_INT);
        $stmt->execute();
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        if ($row) {
            $row['telefono'] = $this->decodeJson($row['telefono']);
            $row['celular'] = $this->decodeJson($row['celular']);
            $row['email'] = $this->decodeJson($row['email']);
        }
        return $row ?: null;
    }

    public function create(int $proveeId, array $data) {
        // IMPORTANTE: user_id es obligatorio y debe ser pasado desde el controlador
        if (!isset($data['user_id'])) {
            throw new InvalidArgumentException('user_id es requerido para crear un contacto de proveedor.');
        }
        
        $sql = "INSERT INTO {$this->table} (provee_id,nombre,apellido,cargo,telefono,celular,email,cumpleanios,origen,estado,user_id)
                VALUES (:provee_id,:nombre,:apellido,:cargo,:telefono,:celular,:email,:cumpleanios,:origen,:estado,:user_id)";
        $stmt = $this->conn->prepare($sql);
        $stmt->bindValue(':provee_id', $proveeId, PDO::PARAM_INT);
        $stmt->bindValue(':nombre', $data['nombre'] ?? '', PDO::PARAM_STR);
        $stmt->bindValue(':apellido', $data['apellido'] ?? null, PDO::PARAM_STR);
        $stmt->bindValue(':cargo', $data['cargo'] ?? null, PDO::PARAM_STR);
        $stmt->bindValue(':telefono', $this->encodeJson($data['telefono'] ?? []), PDO::PARAM_STR);
        $stmt->bindValue(':celular', $this->encodeJson($data['celular'] ?? []), PDO::PARAM_STR);
        $stmt->bindValue(':email', $this->encodeJson($data['email'] ?? []), PDO::PARAM_STR);
        $stmt->bindValue(':cumpleanios', $data['cumpleanios'] ?? null, PDO::PARAM_STR);
        $stmt->bindValue(':origen', $data['origen'] ?? null, PDO::PARAM_STR);
        $stmt->bindValue(':estado', isset($data['estado']) ? (int)$data['estado'] : 1, PDO::PARAM_INT);
        $stmt->bindValue(':user_id', $data['user_id'], PDO::PARAM_INT);
        $stmt->execute();
        return (int)$this->conn->lastInsertId();
    }

    /**
     * Actualizar contacto
     */
    public function update(int $id, array $data) {
        $fields = [];
        $params = [':id' => $id];
        
        $updateable = ['nombre', 'apellido', 'cargo', 'cumpleanios', 'origen', 'user_id'];
        foreach ($updateable as $field) {
            if (array_key_exists($field, $data)) {
                $fields[] = $field . ' = :' . $field;
                $params[':' . $field] = $data[$field] ?? null;
            }
        }
        
        // Campos JSON
        foreach (['telefono', 'celular', 'email'] as $jsonField) {
            if (array_key_exists($jsonField, $data)) {
                $fields[] = $jsonField . ' = :' . $jsonField;
                $params[':' . $jsonField] = $this->encodeJson($data[$jsonField] ?? []);
            }
        }

        // Permitir actualizar el proveedor asociado si se envía
        if (array_key_exists('provee_id', $data)) {
            $fields[] = 'provee_id = :provee_id';
            $params[':provee_id'] = (int)($data['provee_id'] ?? 0);
        }
        
        if (empty($fields)) return false;
        
        $sql = 'UPDATE ' . $this->table . ' SET ' . implode(',', $fields) . ' WHERE idcont_provee = :id';
        $stmt = $this->conn->prepare($sql);
        foreach ($params as $k => $v) {
            if ($k === ':provee_id' || $k === ':id') {
                $stmt->bindValue($k, (int)$v, PDO::PARAM_INT);
            } else {
                $stmt->bindValue($k, $v);
            }
        }
        return $stmt->execute();
    }

    /**
     * Eliminar contacto (soft delete: cambiar estado a 0)
     */
    public function delete(int $id) {
        $stmt = $this->conn->prepare("UPDATE {$this->table} SET estado = 0 WHERE idcont_provee = :id");
        $stmt->bindValue(':id', $id, PDO::PARAM_INT);
        return $stmt->execute();
    }

    private function encodeJson($value) {
        if (is_array($value)) return json_encode(array_values($value), JSON_UNESCAPED_UNICODE);
        if (is_string($value)) return $value;
        return json_encode([], JSON_UNESCAPED_UNICODE);
    }

    private function decodeJson($value) {
        if ($value === null || $value === '') return [];
        $decoded = json_decode($value, true);
        return is_array($decoded) ? $decoded : [];
    }
}
