<?php
class ContactoModel
{
    private PDO $conn;
    private string $table = 'contacto';

    public function __construct(PDO $db)
    {
        $this->conn = $db;
    }

    public function list(array $filters = [], int $limit = 25, int $offset = 0): array
    {
        $sql = "SELECT c.id, c.empresa_id, c.nombres, c.apellidos, c.cargo,
                       c.num_doc, c.tipo_contacto, c.direccion, c.referencia, c.origen,
                       ct_main.telefono AS celular,
                       ce_main.email AS email,
                       (SELECT GROUP_CONCAT(ce.email ORDER BY ce.es_principal DESC, ce.id ASC SEPARATOR ', ')
                        FROM contacto_emails ce
                        WHERE ce.contacto_id = c.id AND ce.estado = 1) AS emails_all,
                       (SELECT COALESCE(NULLIF(TRIM(CONCAT(u.nombres, ' ', u.apellidos)), ''), u.usuario)
                        FROM contacto_historial h
                        LEFT JOIN usuarios u ON u.idusuario = h.usuario_id
                        WHERE h.contacto_id = c.id
                        ORDER BY h.fecha DESC, h.id DESC
                        LIMIT 1) AS usuario_ultimo,
                       c.updated_at,
                       e.razon_social, e.nombre_comercial, e.ruc
                FROM {$this->table} c
                LEFT JOIN empresa e ON e.id = c.empresa_id
                LEFT JOIN contacto_telefonos ct_main ON ct_main.contacto_id = c.id AND ct_main.estado = 1 AND ct_main.es_principal = 1
                LEFT JOIN contacto_emails ce_main ON ce_main.contacto_id = c.id AND ce_main.estado = 1 AND ce_main.es_principal = 1
                WHERE c.estado = 1";
        $params = [];

        $q = trim((string)($filters['q'] ?? ''));
        $campo = trim((string)($filters['campo'] ?? 'todos'));
        if ($q !== '') {
            $like = '%' . $q . '%';
            switch ($campo) {
                case 'id':
                    if (ctype_digit($q)) {
                        $sql .= " AND c.id = :id";
                        $params[':id'] = (int)$q;
                    }
                    break;
                case 'empresa':
                    $sql .= " AND (e.razon_social LIKE :q OR e.nombre_comercial LIKE :q OR e.ruc LIKE :q)";
                    $params[':q'] = $like;
                    break;
                case 'nombres':
                    $sql .= " AND c.nombres LIKE :q";
                    $params[':q'] = $like;
                    break;
                case 'apellidos':
                    $sql .= " AND c.apellidos LIKE :q";
                    $params[':q'] = $like;
                    break;
                case 'cargo':
                    $sql .= " AND c.cargo LIKE :q";
                    $params[':q'] = $like;
                    break;
                case 'celular':
                    $sql .= " AND ct_main.telefono LIKE :q";
                    $params[':q'] = $like;
                    break;
                case 'email':
                    $sql .= " AND ce_main.email LIKE :q";
                    $params[':q'] = $like;
                    break;
                case 'todos':
                default:
                    $sql .= " AND (c.nombres LIKE :q OR c.apellidos LIKE :q OR c.cargo LIKE :q
                               OR ct_main.telefono LIKE :q OR ce_main.email LIKE :q
                               OR e.razon_social LIKE :q OR e.nombre_comercial LIKE :q OR e.ruc LIKE :q)";
                    $params[':q'] = $like;
                    break;
            }
        }

        $sql .= " ORDER BY c.updated_at DESC, c.id DESC LIMIT :limit OFFSET :offset";
        $stmt = $this->conn->prepare($sql);
        foreach ($params as $k => $v) {
            $stmt->bindValue($k, $v, is_int($v) ? PDO::PARAM_INT : PDO::PARAM_STR);
        }
        $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
        $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function count(array $filters = []): int
    {
        $sql = "SELECT COUNT(*)
                FROM {$this->table} c
                LEFT JOIN empresa e ON e.id = c.empresa_id
                LEFT JOIN contacto_telefonos ct_main ON ct_main.contacto_id = c.id AND ct_main.estado = 1 AND ct_main.es_principal = 1
                LEFT JOIN contacto_emails ce_main ON ce_main.contacto_id = c.id AND ce_main.estado = 1 AND ce_main.es_principal = 1
                WHERE c.estado = 1";
        $params = [];

        $q = trim((string)($filters['q'] ?? ''));
        $campo = trim((string)($filters['campo'] ?? 'todos'));
        if ($q !== '') {
            $like = '%' . $q . '%';
            switch ($campo) {
                case 'id':
                    if (ctype_digit($q)) {
                        $sql .= " AND c.id = :id";
                        $params[':id'] = (int)$q;
                    }
                    break;
                case 'empresa':
                    $sql .= " AND (e.razon_social LIKE :q OR e.nombre_comercial LIKE :q OR e.ruc LIKE :q)";
                    $params[':q'] = $like;
                    break;
                case 'nombres':
                    $sql .= " AND c.nombres LIKE :q";
                    $params[':q'] = $like;
                    break;
                case 'apellidos':
                    $sql .= " AND c.apellidos LIKE :q";
                    $params[':q'] = $like;
                    break;
                case 'cargo':
                    $sql .= " AND c.cargo LIKE :q";
                    $params[':q'] = $like;
                    break;
                case 'celular':
                    $sql .= " AND ct_main.telefono LIKE :q";
                    $params[':q'] = $like;
                    break;
                case 'email':
                    $sql .= " AND ce_main.email LIKE :q";
                    $params[':q'] = $like;
                    break;
                case 'todos':
                default:
                    $sql .= " AND (c.nombres LIKE :q OR c.apellidos LIKE :q OR c.cargo LIKE :q
                               OR ct_main.telefono LIKE :q OR ce_main.email LIKE :q
                               OR e.razon_social LIKE :q OR e.nombre_comercial LIKE :q OR e.ruc LIKE :q)";
                    $params[':q'] = $like;
                    break;
            }
        }

        $stmt = $this->conn->prepare($sql);
        foreach ($params as $k => $v) {
            $stmt->bindValue($k, $v, is_int($v) ? PDO::PARAM_INT : PDO::PARAM_STR);
        }
        $stmt->execute();
        return (int)$stmt->fetchColumn();
    }

    public function get(int $id): ?array
    {
        $stmt = $this->conn->prepare(
            "SELECT c.*, e.razon_social, e.nombre_comercial, e.ruc
             FROM {$this->table} c
             LEFT JOIN empresa e ON e.id = c.empresa_id
             WHERE c.id = :id LIMIT 1"
        );
        $stmt->bindValue(':id', $id, PDO::PARAM_INT);
        $stmt->execute();
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        return $row ?: null;
    }

    public function create(array $data): int
    {
        $fields = $this->normalizePayload($data);
        $fields['created_at'] = date('Y-m-d H:i:s');
        $fields['updated_at'] = $fields['created_at'];
        $fields['estado'] = 1;

        $columns = array_keys($fields);
        $placeholders = array_map(fn($c) => ':' . $c, $columns);
        $sql = "INSERT INTO {$this->table} (" . implode(',', $columns) . ") VALUES (" . implode(',', $placeholders) . ")";
        $stmt = $this->conn->prepare($sql);
        foreach ($fields as $k => $v) {
            $stmt->bindValue(':' . $k, $v);
        }
        $stmt->execute();
        return (int)$this->conn->lastInsertId();
    }

    public function update(int $id, array $data): bool
    {
        $fields = $this->normalizePayload($data);
        $fields['updated_at'] = date('Y-m-d H:i:s');
        $sets = [];
        foreach ($fields as $k => $v) {
            $sets[] = $k . ' = :' . $k;
        }
        $sql = "UPDATE {$this->table} SET " . implode(', ', $sets) . " WHERE id = :id";
        $stmt = $this->conn->prepare($sql);
        $stmt->bindValue(':id', $id, PDO::PARAM_INT);
        foreach ($fields as $k => $v) {
            $stmt->bindValue(':' . $k, $v);
        }
        return $stmt->execute();
    }

    public function delete(int $id): bool
    {
        $sql = "UPDATE {$this->table} SET estado = 0, deleted_at = :deleted_at WHERE id = :id";
        $stmt = $this->conn->prepare($sql);
        $stmt->bindValue(':id', $id, PDO::PARAM_INT);
        $stmt->bindValue(':deleted_at', date('Y-m-d H:i:s'));
        return $stmt->execute();
    }

    public function listEmpresas(string $q = ''): array
    {
        $sql = "SELECT id, ruc, razon_social, nombre_comercial
                FROM empresa
                WHERE estado = 1";
        $params = [];
        if ($q !== '') {
            $sql .= " AND (razon_social LIKE :q OR nombre_comercial LIKE :q OR ruc LIKE :q)";
            $params[':q'] = '%' . $q . '%';
        }
        $limit = ($q === '') ? 50 : 200;
        $sql .= " ORDER BY razon_social ASC LIMIT " . (int)$limit;
        $stmt = $this->conn->prepare($sql);
        foreach ($params as $k => $v) {
            $stmt->bindValue($k, $v);
        }
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    private function normalizePayload(array $data): array
    {
        return [
            'empresa_id' => (int)($data['empresa_id'] ?? 0),
            'nombres' => trim((string)($data['nombres'] ?? '')),
            'apellidos' => trim((string)($data['apellidos'] ?? '')),
            'num_doc' => trim((string)($data['num_doc'] ?? '')),
            'cargo' => trim((string)($data['cargo'] ?? '')),
            'origen' => trim((string)($data['origen'] ?? '')),
            'tipo_contacto' => trim((string)($data['tipo_contacto'] ?? '')),
            'fecha_cumpleanos' => $this->normalizeDate($data['fecha_cumpleanos'] ?? null),
            'direccion' => trim((string)($data['direccion'] ?? '')),
            'referencia' => trim((string)($data['referencia'] ?? '')),
        ];
    }

    private function normalizeDate($value): ?string
    {
        $value = trim((string)$value);
        if ($value === '') {
            return null;
        }
        $time = strtotime($value);
        if ($time === false) {
            return null;
        }
        return date('Y-m-d', $time);
    }
}
