<?php
class ContactoEmpresaModel
{
    private PDO $conn;
    private string $table = 'contacto';

    public function __construct(PDO $db)
    {
        $this->conn = $db;
    }

    public function listByEmpresa(int $empresaId): array
    {
        $stmt = $this->conn->prepare(
            "SELECT c.id, c.nombres, c.apellidos,
                    ct_main.telefono AS celular,
                    ce_main.email AS email
             FROM {$this->table} c
             LEFT JOIN contacto_telefonos ct_main ON ct_main.contacto_id = c.id AND ct_main.estado = 1 AND ct_main.es_principal = 1
             LEFT JOIN contacto_emails ce_main ON ce_main.contacto_id = c.id AND ce_main.estado = 1 AND ce_main.es_principal = 1
             WHERE c.empresa_id = :empresa_id AND c.estado = 1
             ORDER BY c.id ASC"
        );
        $stmt->bindValue(':empresa_id', $empresaId, PDO::PARAM_INT);
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function get(int $id): ?array
    {
        $stmt = $this->conn->prepare("SELECT * FROM {$this->table} WHERE id = :id LIMIT 1");
        $stmt->bindValue(':id', $id, PDO::PARAM_INT);
        $stmt->execute();
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        return $row ?: null;
    }

    public function create(int $empresaId, array $data): int
    {
        $fields = $this->normalizePayload($data);
        $fields['empresa_id'] = $empresaId;
        $fields['estado'] = 1;
        $fields['created_at'] = date('Y-m-d H:i:s');
        $fields['updated_at'] = $fields['created_at'];

        $columns = array_keys($fields);
        $placeholders = array_map(fn($c) => ':' . $c, $columns);
        $sql = "INSERT INTO {$this->table} (" . implode(',', $columns) . ") VALUES (" . implode(',', $placeholders) . ")";
        $stmt = $this->conn->prepare($sql);
        foreach ($fields as $k => $v) {
            $stmt->bindValue(':' . $k, $v);
        }
        $stmt->execute();
        return (int)$this->conn->lastInsertId();
    }

    public function update(int $id, array $data): bool
    {
        $fields = $this->normalizePayload($data);
        $fields['updated_at'] = date('Y-m-d H:i:s');
        $sets = [];
        foreach ($fields as $k => $v) {
            $sets[] = $k . ' = :' . $k;
        }
        $sql = "UPDATE {$this->table} SET " . implode(', ', $sets) . " WHERE id = :id";
        $stmt = $this->conn->prepare($sql);
        $stmt->bindValue(':id', $id, PDO::PARAM_INT);
        foreach ($fields as $k => $v) {
            $stmt->bindValue(':' . $k, $v);
        }
        return $stmt->execute();
    }

    public function delete(int $id): bool
    {
        $sql = "UPDATE {$this->table} SET estado = 0, deleted_at = :deleted_at WHERE id = :id";
        $stmt = $this->conn->prepare($sql);
        $stmt->bindValue(':id', $id, PDO::PARAM_INT);
        $stmt->bindValue(':deleted_at', date('Y-m-d H:i:s'));
        return $stmt->execute();
    }

    private function normalizePayload(array $data): array
    {
        return [
            'nombres' => trim((string)($data['nombres'] ?? '')),
            'apellidos' => trim((string)($data['apellidos'] ?? '')),
            'num_doc' => trim((string)($data['num_doc'] ?? '')),
            'cargo' => trim((string)($data['cargo'] ?? '')),
            'origen' => trim((string)($data['origen'] ?? '')),
            'tipo_contacto' => trim((string)($data['tipo_contacto'] ?? '')),
            'fecha_cumpleanos' => $this->normalizeDate($data['fecha_cumpleanos'] ?? null),
            'direccion' => trim((string)($data['direccion'] ?? '')),
            'referencia' => trim((string)($data['referencia'] ?? '')),
        ];
    }

    private function normalizeDate($value): ?string
    {
        $value = trim((string)$value);
        if ($value === '') {
            return null;
        }
        $time = strtotime($value);
        if ($time === false) {
            return null;
        }
        return date('Y-m-d', $time);
    }
}
