<?php
class ContactoEmailModel
{
    private PDO $conn;
    private string $table = 'contacto_emails';

    public function __construct(PDO $db)
    {
        $this->conn = $db;
    }

    public function listByContacto(int $contactoId): array
    {
        $stmt = $this->conn->prepare(
            "SELECT id, email, tipo, es_principal
             FROM {$this->table}
             WHERE contacto_id = :contacto_id AND estado = 1
             ORDER BY es_principal DESC, id ASC"
        );
        $stmt->bindValue(':contacto_id', $contactoId, PDO::PARAM_INT);
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function sync(int $contactoId, array $emails): void
    {
        $normalized = $this->normalizeEmails($emails);
        $existing = $this->fetchExisting($contactoId);
        $now = date('Y-m-d H:i:s');

        $activeEmails = [];
        foreach ($existing as $row) {
            if ((int)$row['estado'] === 1) {
                $activeEmails[$row['email']] = $row;
            }
        }

        $inputEmails = [];
        foreach ($normalized as $row) {
            $inputEmails[$row['email']] = $row;
        }

        foreach ($activeEmails as $email => $row) {
            if (!isset($inputEmails[$email])) {
                $this->setEstado($row['id'], 0, $now);
            }
        }

        foreach ($inputEmails as $email => $row) {
            if (isset($existing[$email])) {
                $this->updateRow($existing[$email]['id'], $row, $now);
            } else {
                $this->insertRow($contactoId, $row, $now);
            }
        }
    }

    private function normalizeEmails(array $emails): array
    {
        $allowedTipos = ['trabajo', 'personal', 'casa', 'otro'];
        $out = [];
        foreach ($emails as $item) {
            if (is_string($item)) {
                $email = strtolower(trim($item));
                $tipo = 'otro';
                $principal = 0;
            } else {
                $email = strtolower(trim((string)($item['email'] ?? '')));
                $tipo = strtolower(trim((string)($item['tipo'] ?? 'otro')));
                $principal = !empty($item['es_principal']) ? 1 : 0;
            }
            if ($email === '') {
                continue;
            }
            if (!in_array($tipo, $allowedTipos, true)) {
                $tipo = 'otro';
            }
            if (!isset($out[$email]) || $principal === 1 || $out[$email]['es_principal'] !== 1) {
                $out[$email] = ['email' => $email, 'tipo' => $tipo, 'es_principal' => $principal];
            }
        }
        return array_values($out);
    }

    private function fetchExisting(int $contactoId): array
    {
        $stmt = $this->conn->prepare(
            "SELECT id, email, tipo, es_principal, estado
             FROM {$this->table}
             WHERE contacto_id = :contacto_id"
        );
        $stmt->execute([':contacto_id' => $contactoId]);
        $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);
        $byEmail = [];
        foreach ($rows as $row) {
            $email = $row['email'];
            if (!isset($byEmail[$email]) || ((int)$byEmail[$email]['estado'] !== 1 && (int)$row['estado'] === 1)) {
                $byEmail[$email] = $row;
            }
        }
        return $byEmail;
    }

    private function setEstado(int $id, int $estado, string $now): void
    {
        $stmt = $this->conn->prepare(
            "UPDATE {$this->table}
             SET estado = :estado, updated_at = :updated_at, es_principal = 0
             WHERE id = :id"
        );
        $stmt->execute([
            ':estado' => $estado,
            ':updated_at' => $now,
            ':id' => $id,
        ]);
    }

    private function updateRow(int $id, array $row, string $now): void
    {
        $stmt = $this->conn->prepare(
            "UPDATE {$this->table}
             SET tipo = :tipo, es_principal = :es_principal, estado = 1, updated_at = :updated_at
             WHERE id = :id"
        );
        $stmt->execute([
            ':tipo' => $row['tipo'],
            ':es_principal' => $row['es_principal'],
            ':updated_at' => $now,
            ':id' => $id,
        ]);
    }

    private function insertRow(int $contactoId, array $row, string $now): void
    {
        $stmt = $this->conn->prepare(
            "INSERT INTO {$this->table} (contacto_id, email, tipo, es_principal, estado, created_at, updated_at)
             VALUES (:contacto_id, :email, :tipo, :es_principal, 1, :created_at, :updated_at)"
        );
        $stmt->execute([
            ':contacto_id' => $contactoId,
            ':email' => $row['email'],
            ':tipo' => $row['tipo'],
            ':es_principal' => $row['es_principal'],
            ':created_at' => $now,
            ':updated_at' => $now,
        ]);
    }
}
