<?php
class CategoriaModel {
    private $conn;
    private $table = 'categoria';

    public function __construct(PDO $db) {
        $this->conn = $db;
    }

    public function list(string $q = ''): array {
        $sql = "SELECT * FROM {$this->table} WHERE status = 1";
        $params = [];
        if ($q !== '') {
            $sql .= " AND (name_cat LIKE :q OR descripcion LIKE :q)";
            $params[':q'] = '%' . $q . '%';
        }
        $sql .= " ORDER BY dateupdate DESC, datecreate DESC, idcat DESC";
        $stmt = $this->conn->prepare($sql);
        foreach ($params as $k => $v) {
            $stmt->bindValue($k, $v, PDO::PARAM_STR);
        }
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function get(int $id): ?array {
        $stmt = $this->conn->prepare("SELECT * FROM {$this->table} WHERE idcat = :id LIMIT 1");
        $stmt->bindValue(':id', $id, PDO::PARAM_INT);
        $stmt->execute();
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        return $row ?: null;
    }

    public function create(array $data): int {
        if (!isset($data['name_cat']) || trim((string)$data['name_cat']) === '') {
            throw new InvalidArgumentException('El nombre de la categoría es obligatorio.');
        }
        $columns = ['name_cat','descripcion','status'];
        $fields = [];
        $placeholders = [];
        $params = [];
        foreach ($columns as $c) {
            if (array_key_exists($c, $data)) {
                $fields[] = $c;
                $placeholders[] = ':' . $c;
                $params[':' . $c] = $data[$c];
            }
        }
        if (empty($fields)) {
            throw new InvalidArgumentException('No hay campos para insertar.');
        }
        $sql = 'INSERT INTO ' . $this->table . ' (' . implode(',', $fields) . ') VALUES (' . implode(',', $placeholders) . ')';
        $stmt = $this->conn->prepare($sql);
        foreach ($params as $k => $v) {
            $stmt->bindValue($k, $v);
        }
        $stmt->execute();
        return (int)$this->conn->lastInsertId();
    }

    public function update(int $id, array $data): bool {
        $columns = ['name_cat','descripcion','status'];
        $sets = [];
        $params = [':id' => $id];
        foreach ($columns as $c) {
            if (array_key_exists($c, $data)) {
                $sets[] = $c . ' = :' . $c;
                $params[':' . $c] = $data[$c];
            }
        }
        if (empty($sets)) {
            return false;
        }
        $sql = 'UPDATE ' . $this->table . ' SET ' . implode(',', $sets) . ' WHERE idcat = :id';
        $stmt = $this->conn->prepare($sql);
        foreach ($params as $k => $v) {
            $stmt->bindValue($k, $v);
        }
        return $stmt->execute();
    }

    public function delete(int $id): bool {
        $stmt = $this->conn->prepare("UPDATE {$this->table} SET status = 0 WHERE idcat = :id");
        $stmt->bindValue(':id', $id, PDO::PARAM_INT);
        return $stmt->execute();
    }
}

