<?php

class CampaniaContacto {

    protected $conn;

    public function __construct($db) {
        $this->conn = $db;
    }

    /* ===================== ASIGNAR ===================== */

    public function asignar(int $campaniaId, int $contactoId): bool {
        $sql = "
            INSERT INTO campania_contacto (campania_id, contacto_id)
            VALUES (:campania_id, :contacto_id)
            ON DUPLICATE KEY UPDATE campania_id = campania_id
        ";

        $stmt = $this->conn->prepare($sql);
        return $stmt->execute([
            ':campania_id' => $campaniaId,
            ':contacto_id' => $contactoId
        ]);
    }

    /* ===================== DESASIGNAR ===================== */

    public function desasignar(int $campaniaId, int $contactoId): bool {
        $sql = "
            DELETE FROM campania_contacto
            WHERE campania_id = :campania_id
              AND contacto_id = :contacto_id
        ";

        $stmt = $this->conn->prepare($sql);
        return $stmt->execute([
            ':campania_id' => $campaniaId,
            ':contacto_id' => $contactoId
        ]);
    }

    /* ===================== CONTACTOS POR CAMPAÑA ===================== */

    public function obtenerContactosPorCampania(int $campaniaId): array {
        $sql = "
            SELECT 
                c.id,
                c.nombres,
                c.apellidos,
                c.cargo,
                COALESCE(ce_evt.email, ce_main.email) AS email,
                COALESCE(ct_fijo.telefono, '') AS telefono,
                COALESCE(ct_movil.telefono, c.celular, '') AS celular,
                e.razon_social AS empresa,
                e.n_trabajadores AS n_trabajadores,
                COALESCE(evt.open, 'No') AS open,
                CASE WHEN evt.open = 'Yes' THEN 1 ELSE 0 END AS open_flag,
                COALESCE(evt.clicks, 0) AS clicks,
                evt.email_subject AS email_subject
            FROM contacto c
            LEFT JOIN empresa e
                ON e.id = c.empresa_id
            INNER JOIN campania_contacto cc 
                ON c.id = cc.contacto_id
            LEFT JOIN contacto_telefonos ct_movil
                ON ct_movil.contacto_id = c.id AND ct_movil.estado = 1 AND ct_movil.es_principal = 1
            LEFT JOIN contacto_telefonos ct_fijo
                ON ct_fijo.id = (
                    SELECT t.id
                    FROM contacto_telefonos t
                    WHERE t.contacto_id = c.id
                      AND t.estado = 1
                      AND t.tipo = 'fijo'
                    ORDER BY t.es_principal DESC, t.id ASC
                    LIMIT 1
                )
            LEFT JOIN campania_contacto_eventos evt
                ON evt.campania_id = cc.campania_id AND evt.contacto_id = c.id
            LEFT JOIN contacto_emails ce_evt
                ON ce_evt.id = evt.contacto_email_id
            LEFT JOIN contacto_emails ce_main
                ON ce_main.contacto_id = c.id AND ce_main.estado = 1 AND ce_main.es_principal = 1
            WHERE cc.campania_id = :campania_id
            ORDER BY c.nombres ASC
        ";

        $stmt = $this->conn->prepare($sql);
        $stmt->execute([':campania_id' => $campaniaId]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    /* ===================== CAMPAÑAS POR CONTACTO ===================== */

    public function obtenerCampanasPorContacto(int $contactoId): array {
        $sql = "
            SELECT 
                cp.id_campania,
                cp.titulo,
                cp.asunto,
                cp.fecha_inicio,
                cp.fecha_fin
            FROM campanias cp
            INNER JOIN campania_contacto cc 
                ON cp.id_campania = cc.campania_id
            WHERE cc.contacto_id = :contacto_id
            ORDER BY cp.created_at DESC
        ";

        $stmt = $this->conn->prepare($sql);
        $stmt->execute([':contacto_id' => $contactoId]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function obtenerPendientesPorCampania(int $campaniaId): array {
        $sql = "
            SELECT
                mep.email,
                mep.mautic_lead_id,
                mep.first_seen_at,
                mep.last_seen_at,
                mep.source
            FROM mautic_email_pendiente mep
            WHERE mep.campania_id = :campania_id
            ORDER BY mep.last_seen_at DESC
        ";

        $stmt = $this->conn->prepare($sql);
        $stmt->execute([':campania_id' => $campaniaId]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
}
