<?php
header('Content-Type: application/json; charset=utf-8');
date_default_timezone_set('America/Lima');

require_once __DIR__ . '/../../config/auth.php';
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../../models/RequerimientoV2Model.php';
require_once __DIR__ . '/../../models/HistoriaActividadRequerimientoModel.php';
require_once __DIR__ . '/../../models/RequerimientoInteraccionModel.php';
require_once __DIR__ . '/../../models/RequerimientoCotizacionModel.php';
require_once __DIR__ . '/../../models/HistorialStatusModel.php';

requireLogin();
requireModuleAccess('requerimientos_v2');

try {
    $db = (new Database())->getConnection();
    $requerimientoModel = new RequerimientoV2Model($db);
    $historiaModel = new HistoriaActividadRequerimientoModel($db);
    $interaccionModel = new RequerimientoInteraccionModel($db);
    $cotizacionModel = new RequerimientoCotizacionModel($db);
    $statusModel = new HistorialStatusModel($db);
} catch (Throwable $e) {
    http_response_code(500);
    echo json_encode(['success' => false, 'message' => 'Error de conexion.']);
    exit();
}

$action = $_REQUEST['action'] ?? 'list';
$usuarioId = (int)($_SESSION['idusuario'] ?? 0);

try {
    switch ($action) {
        case 'list': {
            $term = trim((string)($_GET['q'] ?? ''));
            $filters = [
                'q'          => $term,
                'empresa_id' => $_GET['empresa_id'] ?? null,
                'lead_id'    => $_GET['lead_id'] ?? null,
                'etapa'      => $_GET['etapa'] ?? null,
                'estado'     => $_GET['estado'] ?? null,
            ];
            $page = max(1, (int)($_GET['page'] ?? 1));
            $limit = max(1, (int)($_GET['limit'] ?? 25));
            $offset = ($page - 1) * $limit;

            $rows = $requerimientoModel->list($filters, $limit, $offset);
            $total = $requerimientoModel->count($filters);
            echo json_encode(['success' => true, 'data' => $rows, 'total' => $total]);
            break;
        }

        case 'get': {
            $id = (int)($_GET['id'] ?? 0);
            if ($id <= 0) {
                throw new InvalidArgumentException('ID invalido');
            }
            $row = $requerimientoModel->get($id);
            if (!$row) {
                echo json_encode(['success' => false, 'message' => 'No encontrado']);
                break;
            }
            $withHistoria = !empty($_GET['historia']);
            $withInteracciones = !empty($_GET['interacciones']);
            $response = ['success' => true, 'data' => $row];
            if ($withHistoria) {
                $response['historia'] = $historiaModel->listByRequerimiento($id);
            }
            if ($withInteracciones) {
                $response['interacciones'] = $interaccionModel->listByRequerimiento($id);
            }
            echo json_encode($response);
            break;
        }

        case 'create': {
            $input = json_decode(file_get_contents('php://input'), true);
            if (!is_array($input)) {
                $input = $_POST;
            }
            $id = $requerimientoModel->create($input);
            if ($usuarioId > 0) {
                $historiaModel->add($id, 'creacion', 'Creado manualmente', null, $input['etapa'] ?? 'Pendiente Atender', $usuarioId);
            }
            echo json_encode(['success' => true, 'id' => $id, 'message' => 'Requerimiento creado']);
            break;
        }

        case 'update': {
            $input = json_decode(file_get_contents('php://input'), true);
            if (!is_array($input)) {
                $input = $_POST;
            }
            $id = (int)($input['id'] ?? 0);
            if ($id <= 0) {
                throw new InvalidArgumentException('ID invalido');
            }
            $before = $requerimientoModel->get($id);
            if (!$before) {
                throw new RuntimeException('Requerimiento no encontrado');
            }

            $allowed = [
                'empresa_id', 'lead_id', 'contacto_id', 'titulo', 'descripcion',
                'monto_estimado', 'monto_real', 'tipo_condicion_proyecto', 'motivo_compra',
                'forma_pago', 'nivel_tentativa', 'categoria_productos', 'usuario_apoyo_id',
                'cotizacion_aprobada', 'escalado_proyecto', 'activo_crm',
                'fecha_requerida', 'etapa', 'origen', 'prioridad', 'estado'
            ];
            $changes = [];
            foreach ($allowed as $field) {
                if (!array_key_exists($field, $input)) {
                    continue;
                }
                $newValue = $input[$field];
                if ($newValue === '') {
                    $newValue = null;
                }
                $oldValue = $before[$field] ?? null;
                if ($oldValue === '') {
                    $oldValue = null;
                }
                $newComparable = is_numeric($newValue) ? (string)$newValue : $newValue;
                $oldComparable = is_numeric($oldValue) ? (string)$oldValue : $oldValue;
                if ($newComparable !== $oldComparable) {
                    $changes[$field] = $newValue;
                }
            }

            if (!$changes) {
                echo json_encode(['success' => true, 'message' => 'Sin cambios']);
                break;
            }

            $requerimientoModel->update($id, $changes);
            if ($usuarioId > 0) {
                $etapaAnterior = $before['etapa'] ?? null;
                $etapaNueva = $changes['etapa'] ?? $etapaAnterior;
                $tipo = ($etapaAnterior !== $etapaNueva) ? 'cambio_etapa' : 'actualizacion';
                $historiaModel->add(
                    $id,
                    $tipo,
                    'Actualizado',
                    $tipo === 'cambio_etapa' ? $etapaAnterior : null,
                    $tipo === 'cambio_etapa' ? $etapaNueva : null,
                    $usuarioId
                );
            }
            echo json_encode(['success' => true, 'message' => 'Requerimiento actualizado']);
            break;
        }

        case 'delete': {
            $input = json_decode(file_get_contents('php://input'), true);
            if (!is_array($input)) {
                $input = $_POST;
            }
            $id = (int)($input['id'] ?? 0);
            if ($id <= 0) {
                throw new InvalidArgumentException('ID invalido');
            }
            $requerimientoModel->delete($id);
            if ($usuarioId > 0) {
                $historiaModel->add($id, 'actualizacion', 'Marcado como eliminado', null, null, $usuarioId);
            }
            echo json_encode(['success' => true, 'message' => 'Requerimiento eliminado']);
            break;
        }

        case 'convert_lead': {
            $input = json_decode(file_get_contents('php://input'), true);
            if (!is_array($input)) {
                $input = $_POST;
            }
            $leadId = (int)($input['lead_id'] ?? 0);
            if ($leadId <= 0) {
                throw new InvalidArgumentException('lead_id es obligatorio');
            }

            // Verificar lead existente y extraer datos base
            $stmt = $db->prepare(
                "SELECT id, empresa_id, contacto_detectado_id, razon_social, descripcion, monto_estimado, fecha_estimada
                 FROM lead WHERE id = :id LIMIT 1"
            );
            $stmt->execute([':id' => $leadId]);
            $lead = $stmt->fetch(PDO::FETCH_ASSOC);
            if (!$lead) {
                throw new RuntimeException('Lead no encontrado');
            }

            $empresaId = (int)($input['empresa_id'] ?? $lead['empresa_id']);
            if ($empresaId <= 0) {
                throw new RuntimeException('El lead no tiene empresa asociada; envia empresa_id');
            }

            $data = [
                'empresa_id'      => $empresaId,
                'lead_id'         => $leadId,
                'contacto_id'     => $input['contacto_id'] ?? $lead['contacto_detectado_id'] ?? null,
                'titulo'          => $input['titulo'] ?? "Requerimiento Lead #{$leadId}",
                'descripcion'     => $input['descripcion'] ?? $lead['descripcion'] ?? null,
                'monto_estimado'  => $input['monto_estimado'] ?? $lead['monto_estimado'] ?? null,
                'fecha_requerida' => $input['fecha_requerida'] ?? $lead['fecha_estimada'] ?? null,
                'etapa'           => $input['etapa'] ?? 'Pendiente Atender',
                'origen'          => $input['origen'] ?? 'convertido de lead',
                'prioridad'       => $input['prioridad'] ?? $lead['prioridad'] ?? null,
                'estado'          => $input['estado'] ?? '1',
            ];

            $id = $requerimientoModel->create($data);
            if ($usuarioId > 0) {
                $historiaModel->add($id, 'creacion', "Creado por conversión de lead #{$leadId}", null, $data['etapa'], $usuarioId);
            }
            echo json_encode(['success' => true, 'id' => $id, 'message' => 'Lead convertido a requerimiento']);
            break;
        }

        case 'list_historia': {
            $requerimientoId = (int)($_GET['requerimiento_id'] ?? 0);
            if ($requerimientoId <= 0) {
                throw new InvalidArgumentException('requerimiento_id invalido');
            }
            $rows = $historiaModel->listByRequerimiento($requerimientoId);
            echo json_encode(['success' => true, 'data' => $rows]);
            break;
        }

        case 'add_historia': {
            $input = json_decode(file_get_contents('php://input'), true);
            if (!is_array($input)) {
                $input = $_POST;
            }
            $requerimientoId = (int)($input['requerimiento_id'] ?? 0);
            $tipo = trim((string)($input['tipo_actividad'] ?? ''));
            if ($requerimientoId <= 0 || $tipo === '') {
                throw new InvalidArgumentException('requerimiento_id y tipo_actividad son obligatorios');
            }
            $historiaModel->add(
                $requerimientoId,
                $tipo,
                $input['detalle'] ?? null,
                $input['etapa_anterior'] ?? null,
                $input['etapa_nueva'] ?? null,
                $usuarioId ?: null
            );
            echo json_encode(['success' => true, 'message' => 'Historial agregado']);
            break;
        }

        case 'list_interacciones': {
            $requerimientoId = (int)($_GET['requerimiento_id'] ?? 0);
            if ($requerimientoId <= 0) {
                throw new InvalidArgumentException('requerimiento_id invalido');
            }
            $rows = $interaccionModel->listByRequerimiento($requerimientoId);
            echo json_encode(['success' => true, 'data' => $rows]);
            break;
        }

        case 'list_historial_status': {
            $contactoId = (int)($_GET['contacto_id'] ?? 0);
            if ($contactoId <= 0) {
                throw new InvalidArgumentException('contacto_id invalido');
            }
            $rows = $statusModel->listByContacto($contactoId);
            echo json_encode(['success' => true, 'data' => $rows]);
            break;
        }

        case 'add_interaccion': {
            $input = json_decode(file_get_contents('php://input'), true);
            if (!is_array($input)) {
                $input = $_POST;
            }
            $input['user_id'] = $usuarioId ?: ($input['user_id'] ?? null);
            $id = $interaccionModel->add($input);
            if ($usuarioId > 0) {
                $historiaModel->add(
                    (int)($input['requerimiento_id'] ?? 0),
                    'interaccion',
                    'Interaccion agregada',
                    null,
                    null,
                    $usuarioId
                );
            }
            echo json_encode(['success' => true, 'id' => $id, 'message' => 'Interaccion guardada']);
            break;
        }

        case 'list_cotizaciones': {
            $requerimientoId = (int)($_GET['requerimiento_id'] ?? 0);
            if ($requerimientoId <= 0) {
                throw new InvalidArgumentException('requerimiento_id invalido');
            }
            $rows = $cotizacionModel->listByRequerimiento($requerimientoId);
            echo json_encode(['success' => true, 'data' => $rows]);
            break;
        }

        case 'add_cotizacion': {
            $requerimientoId = (int)($_POST['requerimiento_id'] ?? 0);
            if ($requerimientoId <= 0) {
                throw new InvalidArgumentException('requerimiento_id es obligatorio');
            }

            if (empty($_FILES['archivo']['name'])) {
                throw new InvalidArgumentException('El archivo PDF es obligatorio');
            }
            if ($_FILES['archivo']['error'] !== UPLOAD_ERR_OK) {
                throw new RuntimeException('Error al recibir el archivo');
            }
            if (empty($_FILES['archivo']['tmp_name']) || !is_uploaded_file($_FILES['archivo']['tmp_name'])) {
                throw new RuntimeException('Archivo temporal invalido');
            }

            $ext = strtolower(pathinfo($_FILES['archivo']['name'], PATHINFO_EXTENSION));
            if ($ext !== 'pdf') {
                throw new RuntimeException('Solo se permite PDF');
            }

            $maxBytes = 3 * 1024 * 1024;
            if (!empty($_FILES['archivo']['size']) && $_FILES['archivo']['size'] > $maxBytes) {
                throw new RuntimeException('El archivo supera el limite admitido');
            }

            $mime = null;
            $finfo = finfo_open(FILEINFO_MIME_TYPE);
            if ($finfo) {
                $mime = finfo_file($finfo, $_FILES['archivo']['tmp_name']);
                finfo_close($finfo);
            }
            if ($mime !== 'application/pdf') {
                throw new RuntimeException('El archivo no es un PDF valido');
            }

            $uploadDir = __DIR__ . '/../../assets/uploads/requerimiento_cotizaciones/';
            if (!is_dir($uploadDir)) {
                mkdir($uploadDir, 0755, true);
            }

            $fileName = 'req_' . $requerimientoId . '_' . date('Ymd_His') . '_' . bin2hex(random_bytes(4)) . '.' . $ext;
            $destino = $uploadDir . $fileName;
            if (!move_uploaded_file($_FILES['archivo']['tmp_name'], $destino)) {
                throw new RuntimeException('No se pudo guardar el archivo');
            }

            $monto = $_POST['monto'] ?? null;
            if ($monto === null || $monto === '') {
                throw new InvalidArgumentException('El monto es obligatorio');
            }

            $data = [
                'requerimiento_id' => $requerimientoId,
                'archivo'          => 'assets/uploads/requerimiento_cotizaciones/' . $fileName,
                'monto'            => $monto,
                'observaciones'    => $_POST['observaciones'] ?? null,
                'user_id'          => $usuarioId ?: null,
            ];
            $id = $cotizacionModel->add($data);
            $requerimientoModel->update($requerimientoId, ['monto_real' => $monto]);
            if ($usuarioId > 0) {
                $historiaModel->add($requerimientoId, 'cotizacion', 'Cotizacion agregada', null, null, $usuarioId);
            }

            echo json_encode([
                'success' => true,
                'id' => $id,
                'monto_real' => $monto,
                'archivo' => $data['archivo'],
            ]);
            break;
        }

        default:
            echo json_encode(['success' => false, 'message' => 'Accion no soportada']);
            break;
    }
} catch (Throwable $e) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'No se pudo procesar la solicitud.']);
}
