<?php
header('Content-Type: application/json; charset=utf-8');
date_default_timezone_set('America/Lima');

require_once __DIR__ . '/../../config/auth.php';
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../../models/ProviderProductModel.php';
require_once __DIR__ . '/../../models/ProveedorModel.php';
require_once __DIR__ . '/../../models/ProductModel.php';

requireLogin();
requireModuleAccess('productos');

try {
    $db = (new Database())->getConnection();
    $providerProduct = new ProviderProductModel($db);
    $proveedor = new ProveedorModel($db);
    $producto = new ProductModel($db);
} catch (Throwable $e) {
    http_response_code(500);
    echo json_encode(['success' => false, 'message' => 'Error de conexión a base de datos: ' . $e->getMessage()]);
    exit();
}

$action = $_REQUEST['action'] ?? 'list';

function pp_json_input(): array
{
    $raw = file_get_contents('php://input');
    if ($raw === false || $raw === '') {
        return $_POST ?? [];
    }
    $data = json_decode($raw, true);
    if (!is_array($data)) {
        return $_POST ?? [];
    }
    return $data;
}

/**
 * Normaliza un campo que se guarda en columna JSON.
 * - Si el texto ya es JSON válido, se guarda tal cual.
 * - Si no, se envuelve como string JSON ("texto...").
 */
function pp_normalize_json_field(string $text): string
{
    $trimmed = trim($text);
    if ($trimmed === '') {
        return json_encode('', JSON_UNESCAPED_UNICODE);
    }
    $decoded = json_decode($trimmed, true);
    if (json_last_error() === JSON_ERROR_NONE) {
        // Ya es JSON válido, lo guardamos como estaba
        return $trimmed;
    }
    // Texto libre: lo guardamos como string JSON
    return json_encode($trimmed, JSON_UNESCAPED_UNICODE);
}

try {
    switch ($action) {
        case 'list': {
            $q = trim((string)($_GET['q'] ?? ''));
            $sql = "SELECT pp.*, 
                           p.razon_social, p.razon_comercial, p.ruc,
                           pm.name_prod, pm.variant_name
                    FROM provider_product pp
                    INNER JOIN proveedores p ON pp.provider_id = p.idprovee
                    INNER JOIN product_model pm ON pp.prodmodel_id = pm.idprodmodel
                    WHERE pp.status = 1";
            $params = [];
            if ($q !== '') {
                $sql .= " AND (pm.name_prod LIKE :q OR pm.variant_name LIKE :q OR p.razon_social LIKE :q OR p.ruc LIKE :q)";
                $params[':q'] = '%' . $q . '%';
            }
            $sql .= " ORDER BY pp.dateupdate DESC, pp.datecreate DESC, pp.idprovprod DESC";
            $stmt = $db->prepare($sql);
            foreach ($params as $k => $v) {
                $stmt->bindValue($k, $v, PDO::PARAM_STR);
            }
            $stmt->execute();
            $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);
            echo json_encode(['success' => true, 'data' => $rows]);
            break;
        }

        case 'get': {
            $id = (int)($_GET['id'] ?? 0);
            if ($id <= 0) {
                throw new InvalidArgumentException('ID de oferta inválido');
            }
            $stmt = $db->prepare("SELECT * FROM provider_product WHERE idprovprod = :id LIMIT 1");
            $stmt->bindValue(':id', $id, PDO::PARAM_INT);
            $stmt->execute();
            $row = $stmt->fetch(PDO::FETCH_ASSOC) ?: null;
            echo json_encode(['success' => (bool)$row, 'data' => $row]);
            break;
        }

        case 'list_proveedores': {
            $filters = [];
            $filters['q'] = trim((string)($_GET['q'] ?? ''));
            $rows = $proveedor->list($filters, 100, 0);
            echo json_encode(['success' => true, 'data' => $rows]);
            break;
        }

        case 'list_productos': {
            $q = trim((string)($_GET['q'] ?? ''));
            $rows = $producto->list($q);
            echo json_encode(['success' => true, 'data' => $rows]);
            break;
        }

        case 'create':
        case 'update': {
            $input = pp_json_input();
            if (!is_array($input)) {
                throw new InvalidArgumentException('Entrada inválida');
            }

            $providerId = (int)($input['provider_id'] ?? 0);
            $prodmodelId = (int)($input['prodmodel_id'] ?? 0);
            if ($providerId <= 0 || $prodmodelId <= 0) {
                throw new InvalidArgumentException('Proveedor y producto son obligatorios.');
            }

            $offerPayload = [
                'provider_id' => $providerId,
                'prodmodel_id' => $prodmodelId,
                'provider_sku' => $input['provider_sku'] ?? null,
            ];

            if (isset($input['price_unit']) && is_array($input['price_unit'])) {
                $offerPayload['price_unit'] = json_encode($input['price_unit'], JSON_UNESCAPED_UNICODE);
            }
            if (isset($input['price_total']) && is_array($input['price_total'])) {
                $offerPayload['price_total'] = json_encode($input['price_total'], JSON_UNESCAPED_UNICODE);
            }
            // provider_detail y provider_specs ahora son LONGTEXT: se guarda texto plano.
            if (array_key_exists('provider_detail', $input)) {
                if (is_string($input['provider_detail']) && $input['provider_detail'] !== '') {
                    $offerPayload['provider_detail'] = $input['provider_detail'];
                } else {
                    $offerPayload['provider_detail'] = null;
                }
            }
            if (array_key_exists('provider_specs', $input)) {
                if (is_string($input['provider_specs']) && $input['provider_specs'] !== '') {
                    $offerPayload['provider_specs'] = $input['provider_specs'];
                } else {
                    $offerPayload['provider_specs'] = null;
                }
            }

            if (array_key_exists('lead_time_days', $input)) {
                $lead = trim((string)$input['lead_time_days']);
                $offerPayload['lead_time_days'] = ($lead !== '') ? $lead : null;
            }

            if (array_key_exists('quote_date', $input)) {
                $qd = trim((string)$input['quote_date']);
                if ($qd !== '') {
                    $today = date('Y-m-d');
                    if ($qd > $today) {
                        throw new InvalidArgumentException('La fecha de cotizacion no puede ser posterior a hoy.');
                    }
                    $offerPayload['quote_date'] = $qd;
                } else {
                    $offerPayload['quote_date'] = null;
                }
            }

            if (isset($input['validity_days'])) {
                $offerPayload['validity_days'] = (int)$input['validity_days'];
            }
            if (isset($input['status'])) {
                $offerPayload['status'] = (int)$input['status'];
            }

            if ($action === 'create') {
                // Si no hay fecha de cotización definida, usar la fecha actual
                if (!empty($input['quote_date'])) {
                    $offerPayload['quote_date'] = $input['quote_date'];
                } else {
                    $offerPayload['quote_date'] = date('Y-m-d');
                }

                $id = $providerProduct->create($offerPayload);
                echo json_encode(['success' => true, 'id' => $id]);
            } else {
                $id = (int)($input['id'] ?? 0);
                if ($id <= 0) {
                    throw new InvalidArgumentException('ID de oferta inválido');
                }
                $sets = [];
                $params = [':id' => $id];
                $columns = [
                    'provider_id',
                    'prodmodel_id',
                    'provider_sku',
                    'price_unit',
                    'price_total',
                    'lead_time_days',
                    'provider_detail',
                    'provider_specs',
                    'quote_date',
                    'validity_days',
                    'status',
                ];
                foreach ($columns as $c) {
                    if (array_key_exists($c, $offerPayload)) {
                        $sets[] = $c . ' = :' . $c;
                        $params[':' . $c] = $offerPayload[$c];
                    }
                }
                if (!empty($sets)) {
                    $sql = 'UPDATE provider_product SET ' . implode(',', $sets) . ' WHERE idprovprod = :id';
                    $stmt = $db->prepare($sql);
                    foreach ($params as $k => $v) {
                        $stmt->bindValue($k, $v);
                    }
                    $ok = $stmt->execute();
                } else {
                    $ok = false;
                }
                echo json_encode(['success' => (bool)$ok]);
            }
            break;
        }

        case 'delete': {
            $id = (int)($_POST['id'] ?? 0);
            if ($id <= 0) {
                throw new InvalidArgumentException('ID de oferta inválido');
            }
            $stmt = $db->prepare('UPDATE provider_product SET status = 0 WHERE idprovprod = :id');
            $stmt->bindValue(':id', $id, PDO::PARAM_INT);
            $ok = $stmt->execute();
            echo json_encode(['success' => (bool)$ok]);
            break;
        }

        default:
            echo json_encode(['success' => false, 'message' => 'Acción no soportada']);
    }
} catch (Throwable $e) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => $e->getMessage()]);
}
