<?php
header('Content-Type: application/json; charset=utf-8');
date_default_timezone_set('America/Lima');

require_once __DIR__ . '/../../config/auth.php';
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../../models/ProveedorModel.php';
require_once __DIR__ . '/../../models/ContactoProveedorModel.php';
require_once __DIR__ . '/../../models/User.php';

requireLogin();
requireModuleAccess('proveedores');

function normalize_tax_identifier(array &$input): void {
    if (!array_key_exists('tax_id_value', $input) && !array_key_exists('ruc', $input)) {
        return;
    }
    $allowed = ['RUC','CUIT','VAT','TIN','NIT','NIF','OTRO'];
    $value = trim((string)($input['tax_id_value'] ?? ($input['ruc'] ?? '')));
    $type = strtoupper(trim((string)($input['tax_id_type'] ?? '')));
    if ($type === '' && isset($input['ruc'])) {
        $current = trim((string)$input['ruc']);
        foreach ($allowed as $candidate) {
            if ($candidate === 'RUC') {
                continue;
            }
            $prefix = $candidate . ' ';
            if (stripos($current, $prefix) === 0) {
                $type = $candidate;
                if ($value === '') {
                    $value = substr($current, strlen($prefix));
                }
                break;
            }
        }
    }
    if (!in_array($type, $allowed, true)) {
        $type = 'RUC';
    }
    if ($value === '') {
        $input['ruc'] = '';
        return;
    }
    $input['ruc'] = ($type === 'RUC') ? $value : ($type . ' ' . $value);
}

try {
    $db = (new Database())->getConnection();
    $proveedor = new ProveedorModel($db);
    $contacto  = new ContactoProveedorModel($db);
} catch (Throwable $e) {
    http_response_code(500);
    echo json_encode(['success' => false, 'message' => 'Error de conexión a base de datos: ' . $e->getMessage()]);
    exit();
}

$action = $_REQUEST['action'] ?? 'list';

try {
    switch ($action) {
        case 'list': {
            $term = trim((string)($_GET['q'] ?? ''));
            $by   = trim((string)($_GET['by'] ?? ''));
            $page = max(1, (int)($_GET['page'] ?? 1));
            $limit = max(1, (int)($_GET['limit'] ?? 25));
            $offset = ($page - 1) * $limit;

            // Compatibilidad: si viene ruc antiguo
            if (isset($_GET['ruc']) && $_GET['ruc'] !== '') {
                $by = 'ruc';
                $term = $_GET['ruc'];
            }

            // Construir búsqueda flexible con JOIN a usuarios para filtrar por usuario
            $baseFrom = ' FROM proveedores p LEFT JOIN usuarios u ON p.user_id = u.idusuario WHERE p.status = 1 ';
            $where = '';
            $params = [];
            if ($term !== '') {
                $like = '%' . $term . '%';
                switch ($by) {
                    case 'id':
                        $where = ' AND p.idprovee = :id ';
                        $params[':id'] = (int)$term;
                        break;
                    case 'ruc':
                        $where = ' AND p.ruc LIKE :t ';
                        $params[':t'] = $like;
                        break;
                    case 'razon_social':
                        $where = ' AND p.razon_social LIKE :t ';
                        $params[':t'] = $like;
                        break;
                    case 'email':
                        $where = ' AND p.email LIKE :t ';
                        $params[':t'] = $like;
                        break;
                    case 'telefono':
                        $where = ' AND (p.telefono LIKE :t OR p.celular LIKE :t) ';
                        $params[':t'] = $like;
                        break;
                    case 'usuario':
                        $where = ' AND (u.nombres LIKE :t OR u.apellidos LIKE :t OR u.usuario LIKE :t) ';
                        $params[':t'] = $like;
                        break;
                    case 'todos':
                    default:
                        $where = ' AND (p.ruc LIKE :t OR p.razon_social LIKE :t OR p.razon_comercial LIKE :t '
                               . ' OR p.email LIKE :t OR p.telefono LIKE :t OR p.celular LIKE :t '
                               . ' OR u.nombres LIKE :t OR u.apellidos LIKE :t OR u.usuario LIKE :t ';
                        // si es numérico, incluir coincidencia exacta por id
                        if (is_numeric($term)) {
                            $where .= ' OR p.idprovee = :idExact ';
                            $params[':idExact'] = (int)$term;
                        }
                        $where .= ') ';
                        $params[':t'] = $like;
                        break;
                }
            }

            // Contar registros de proveedores de forma estable
            $sqlCount = 'SELECT COUNT(*) ' . $baseFrom . $where;
            $stmtC = $db->prepare($sqlCount);
            foreach ($params as $k => $v) $stmtC->bindValue($k, $v, is_int($v) ? PDO::PARAM_INT : PDO::PARAM_STR);
            $stmtC->execute();
            $total = (int)$stmtC->fetchColumn();

            // Orden estable para evitar saltos/dobles entre páginas cuando hay fechas nulas o iguales
            $sql = 'SELECT p.* ' . $baseFrom . $where . ' ORDER BY p.dateupdate DESC, p.datecreated DESC, p.idprovee DESC LIMIT :limit OFFSET :offset';
            $stmt = $db->prepare($sql);
            foreach ($params as $k => $v) $stmt->bindValue($k, $v, is_int($v) ? PDO::PARAM_INT : PDO::PARAM_STR);
            $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
            $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
            $stmt->execute();
            $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);

            // Adjuntar nombre de usuario
            $uModel = new User($db, $_SESSION['idusuario'] ?? 0);
            foreach ($rows as $i => $r) {
                $uid = (int)($r['user_id'] ?? 0);
                $rows[$i]['user_fullname'] = null;
                if ($uid > 0) {
                    try {
                        $uModel->idusuario = $uid;
                        $u = $uModel->readOne();
                        if ($u) $rows[$i]['user_fullname'] = trim(($u->nombres ?? '') . ' ' . ($u->apellidos ?? ''));
                    } catch (Throwable $e) {}
                }
            }
            echo json_encode(['success' => true, 'data' => $rows, 'total' => $total]);
            break;
        }

        case 'get': {
            $id = (int)($_GET['id'] ?? 0);
            if ($id <= 0) throw new InvalidArgumentException('ID inválido');
            $row = $proveedor->get($id);
            if (!$row) { echo json_encode(['success' => false, 'message' => 'No encontrado']); break; }
            $rowsContactos = $contacto->listByProveedor($id);
            $userInfo = null;
            if (!empty($row['user_id'])) {
                try {
                    $uModel = new User($db, $_SESSION['idusuario'] ?? 0);
                    $uModel->idusuario = (int)$row['user_id'];
                    $uLoaded = $uModel->readOne();
                    if ($uLoaded) {
                        $userInfo = ['id' => (int)$uLoaded->idusuario, 'nombres' => $uLoaded->nombres, 'apellidos' => $uLoaded->apellidos];
                    }
                } catch (Throwable $e) { $userInfo = null; }
            }
            echo json_encode(['success' => true, 'data' => $row, 'contactos' => $rowsContactos, 'user' => $userInfo]);
            break;
        }

        case 'create': {
            $input = json_decode(file_get_contents('php://input'), true) ?? $_POST;
            normalize_tax_identifier($input);
            if (empty($input['ruc'])) throw new InvalidArgumentException('El identificador fiscal es obligatorio');
            // Email
            if (isset($input['email']) && $input['email'] !== '' && $input['email'] !== null) {
                $email = trim((string)$input['email']);
                if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
                    echo json_encode(['success' => false, 'message' => 'El correo ingresado no es válido.']);
                    break;
                }
            }
            // Teléfono/Celular
            $pais = strtolower(trim((string)($input['pais'] ?? '')));
            $telefono = isset($input['telefono']) ? (string)$input['telefono'] : '';
            $celular  = isset($input['celular'])  ? (string)$input['celular']  : '';
            $digitsTel = preg_replace('/\D+/', '', $telefono);
            $hasPlusTel = strpos($telefono, '+') !== false;
            $digitsCel = preg_replace('/\D+/', '', $celular);
            $hasPlusCel = strpos($celular, '+') !== false;
            if ($telefono !== '') {
                if ($pais === 'peru' || $pais === 'perú' || $pais === 'pe') {
                    if ($hasPlusTel || strlen($digitsTel) < 6 || strlen($digitsTel) > 15) {
                        echo json_encode(['success' => false, 'message' => 'El teléfono no es válido. Para Perú use solo números (6 a 15 dígitos) sin código de país.']);
                        break;
                    }
                } else {
                    if (strlen($digitsTel) < 6 || strlen($digitsTel) > 15) {
                        echo json_encode(['success' => false, 'message' => 'El teléfono no es válido. Debe tener entre 6 y 15 dígitos (puede incluir código de país).']);
                        break;
                    }
                }
            }
            if ($celular !== '') {
                if ($pais === 'peru' || $pais === 'perú' || $pais === 'pe') {
                    if ($hasPlusCel || strlen($digitsCel) !== 9 || !preg_match('/^9\d{8}$/', $digitsCel)) {
                        echo json_encode(['success' => false, 'message' => 'El celular no es válido. Para Perú debe tener 9 dígitos y empezar con 9, sin código de país.']);
                        break;
                    }
                } else {
                    if (strlen($digitsCel) < 8 || strlen($digitsCel) > 15) {
                        echo json_encode(['success' => false, 'message' => 'El celular no es válido. Debe tener entre 8 y 15 dígitos (puede incluir código de país).']);
                        break;
                    }
                }
            }

            if ($proveedor->getByRuc($input['ruc'])) {
                echo json_encode(['success' => false, 'message' => 'El identificador fiscal ya esta registrado']);
                break;
            }
            $input['user_id'] = $_SESSION['idusuario'] ?? 0;
            if (empty($input['user_id'])) {
                echo json_encode(['success' => false, 'message' => 'Usuario no autenticado.']);
                break;
            }
            $id = $proveedor->create($input);
            echo json_encode(['success' => true, 'id' => $id]);
            break;
        }

        case 'update': {
            $input = json_decode(file_get_contents('php://input'), true);
            if (!is_array($input)) $input = $_POST;
            $id = (int)($input['id'] ?? 0);
            if ($id <= 0) throw new InvalidArgumentException('ID inválido');

            normalize_tax_identifier($input);

            $normalize = function($v){ $v = is_string($v) ? trim($v) : $v; return ($v === '' ? null : $v); };
            foreach (['razon_social','razon_comercial','rubro_sector','tipo_provee','calificacion','telefono','celular','email','aniversario','pais','distrito','ciudad','direccion','Referencia','paginaweb','origen','condicion_contribuyente','estado_contribuyente'] as $k){
                if(array_key_exists($k, $input)) $input[$k] = $normalize($input[$k]);
            }
            foreach (['perfil','catalogos','redes_sociales'] as $jk){
                if(array_key_exists($jk, $input)){
                    $val = $input[$jk];
                    if (is_array($val) || is_object($val)) { $val = json_encode($val, JSON_UNESCAPED_UNICODE); }
                    if (is_string($val) && $val !== '' && json_decode($val, true) === null) {
                        echo json_encode(['success'=>false,'message'=>'Campo '.$jk.' no es JSON válido']);
                        break 2;
                    }
                    $input[$jk] = $val;
                }
            }

            // Email
            if (array_key_exists('email', $input) && $input['email'] !== '' && $input['email'] !== null) {
                $emailUp = trim((string)$input['email']);
                if (!filter_var($emailUp, FILTER_VALIDATE_EMAIL)) {
                    echo json_encode(['success' => false, 'message' => 'El correo ingresado no es válido.']);
                    break;
                }
            }
            // Teléfono/Celular
            $pais = strtolower(trim((string)($input['pais'] ?? '')));
            if (array_key_exists('telefono', $input) && $input['telefono'] !== null && $input['telefono'] !== '') {
                $telefono = (string)$input['telefono'];
                $digitsTel = preg_replace('/\D+/', '', $telefono);
                $hasPlusTel = strpos($telefono, '+') !== false;
                if ($pais === 'peru' || $pais === 'perú' || $pais === 'pe') {
                    if ($hasPlusTel || strlen($digitsTel) < 6 || strlen($digitsTel) > 15) {
                        echo json_encode(['success' => false, 'message' => 'El teléfono no es válido. Para Perú use solo números (6 a 15 dígitos) sin código de país.']);
                        break;
                    }
                } else {
                    if (strlen($digitsTel) < 6 || strlen($digitsTel) > 15) {
                        echo json_encode(['success' => false, 'message' => 'El teléfono no es válido. Debe tener entre 6 y 15 dígitos (puede incluir código de país).']);
                        break;
                    }
                }
            }
            if (array_key_exists('celular', $input) && $input['celular'] !== null && $input['celular'] !== '') {
                $celular = (string)$input['celular'];
                $digitsCel = preg_replace('/\D+/', '', $celular);
                $hasPlusCel = strpos($celular, '+') !== false;
                if ($pais === 'peru' || $pais === 'perú' || $pais === 'pe') {
                    if ($hasPlusCel || strlen($digitsCel) !== 9 || !preg_match('/^9\d{8}$/', $digitsCel)) {
                        echo json_encode(['success' => false, 'message' => 'El celular no es válido. Para Perú debe tener 9 dígitos y empezar con 9, sin código de país.']);
                        break;
                    }
                } else {
                    if (strlen($digitsCel) < 8 || strlen($digitsCel) > 15) {
                        echo json_encode(['success' => false, 'message' => 'El celular no es válido. Debe tener entre 8 y 15 dígitos (puede incluir código de país).']);
                        break;
                    }
                }
            }

            // Asegurar trazabilidad: si hay usuario en sesión, asociarlo
            $currentUserId = (int)($_SESSION['idusuario'] ?? 0);
            if ($currentUserId > 0) {
                $input['user_id'] = $currentUserId;
            }

            $ok = $proveedor->update($id, $input);
            echo json_encode(['success' => (bool)$ok]);
            break;
        }

        case 'delete': {
            $id = (int)($_POST['id'] ?? 0);
            if ($id <= 0) throw new InvalidArgumentException('ID inválido');
            $ok = $proveedor->delete($id);
            echo json_encode(['success' => (bool)$ok]);
            break;
        }

        case 'add_contacto': {
            $input = json_decode(file_get_contents('php://input'), true) ?? $_POST;
            $proveeId = (int)($input['provee_id'] ?? 0);
            if ($proveeId <= 0) throw new InvalidArgumentException('Proveedor inválido');
            $input['user_id'] = $_SESSION['idusuario'] ?? 0;
            if (empty($input['user_id'])) {
                echo json_encode(['success'=>false,'message'=>'Usuario no autenticado.']);
                break;
            }
            $cid = $contacto->create($proveeId, $input);
            echo json_encode(['success' => true, 'id' => $cid]);
            break;
        }

        case 'list_contactos': {
            $proveeId = (int)($_GET['provee_id'] ?? 0);
            if ($proveeId <= 0) throw new InvalidArgumentException('Proveedor inválido');
            $rows = $contacto->listByProveedor($proveeId);
            echo json_encode(['success' => true, 'data' => $rows]);
            break;
        }

        case 'search_ruc': {
            $ruc = trim((string)($_GET['ruc'] ?? ''));
            if ($ruc === '') throw new InvalidArgumentException('RUC requerido');
            $row = $proveedor->getByRuc($ruc);
            echo json_encode(['success' => true, 'local' => $row, 'sunat' => null, 'integration_available' => false]);
            break;
        }

        case 'list_simple': {
            $q = $_GET['q'] ?? '';
            $stmt = $db->prepare(
                "SELECT idprovee, razon_comercial, ruc, razon_social 
                 FROM proveedores 
                 WHERE status = 1 " . (!empty($q) ? "AND (razon_comercial LIKE :q OR ruc LIKE :q OR razon_social LIKE :q)" : "") . "
                 ORDER BY razon_comercial ASC
                 LIMIT 100"
            );
            if (!empty($q)) {
                $stmt->bindValue(':q', '%' . $q . '%', PDO::PARAM_STR);
            }
            $stmt->execute();
            $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);
            echo json_encode(['success' => true, 'data' => $rows]);
            break;
        }

        default:
            echo json_encode(['success' => false, 'message' => 'Acción no soportada']);
    }
} catch (Throwable $e) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => $e->getMessage()]);
}
