<?php
header('Content-Type: application/json; charset=utf-8');
date_default_timezone_set('America/Lima');

require_once __DIR__ . '/../../config/auth.php';
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../../models/ProductModel.php';

requireLogin();
requireModuleAccess('productos');

try {
    $db = (new Database())->getConnection();
    $productModel = new ProductModel($db);
} catch (Throwable $e) {
    http_response_code(500);
    echo json_encode(['success' => false, 'message' => 'Error de conexión a base de datos: ' . $e->getMessage()]);
    exit();
}

$action = $_REQUEST['action'] ?? 'list';

function producto_json_input(): array
{
    $raw = file_get_contents('php://input');
    if ($raw === false || $raw === '') {
        return $_POST ?? [];
    }
    $data = json_decode($raw, true);
    if (!is_array($data)) {
        return $_POST ?? [];
    }
    return $data;
}

try {
    switch ($action) {
        case 'list': {
            $q = trim((string)($_GET['q'] ?? ''));
            $rows = $productModel->list($q);
            echo json_encode(['success' => true, 'data' => $rows]);
            break;
        }

        case 'get': {
            $id = (int)($_GET['id'] ?? 0);
            if ($id <= 0) {
                throw new InvalidArgumentException('ID de producto inválido');
            }
            $row = $productModel->get($id);
            echo json_encode(['success' => (bool)$row, 'data' => $row]);
            break;
        }

        case 'create': {
            $input = producto_json_input();
            if (!is_array($input)) {
                throw new InvalidArgumentException('Entrada inválida');
            }

            $payload = [
                'name_prod' => trim((string)($input['name_prod'] ?? '')),
                'catg_id' => (int)($input['catg_id'] ?? 0),
                'variant_code' => $input['variant_code'] ?? null,
                'variant_name' => $input['variant_name'] ?? null,
                'img' => $input['img'] ?? null,
            ];

            if (isset($input['catg_ids']) && is_array($input['catg_ids'])) {
                $payload['catg_ids'] = json_encode(array_values($input['catg_ids']), JSON_UNESCAPED_UNICODE);
            }

            if (isset($input['detail'])) {
                if (is_array($input['detail'])) {
                    $payload['detail'] = json_encode($input['detail'], JSON_UNESCAPED_UNICODE);
                } elseif (is_string($input['detail']) && $input['detail'] !== '') {
                    // Guardar texto libre como JSON string válido
                    $payload['detail'] = json_encode($input['detail'], JSON_UNESCAPED_UNICODE);
                }
            }

            if (isset($input['status'])) {
                $payload['status'] = (int)$input['status'];
            }

            $id = $productModel->create($payload);
            echo json_encode(['success' => true, 'id' => $id]);
            break;
        }

        case 'update': {
            $input = producto_json_input();
            if (!is_array($input)) {
                throw new InvalidArgumentException('Entrada inválida');
            }

            $id = (int)($input['id'] ?? 0);
            if ($id <= 0) {
                throw new InvalidArgumentException('ID de producto inválido');
            }

            $payload = [];
            if (array_key_exists('name_prod', $input)) {
                $payload['name_prod'] = trim((string)$input['name_prod']);
            }
            if (array_key_exists('catg_id', $input)) {
                $payload['catg_id'] = (int)$input['catg_id'];
            }
            if (array_key_exists('variant_code', $input)) {
                $payload['variant_code'] = $input['variant_code'];
            }
            if (array_key_exists('variant_name', $input)) {
                $payload['variant_name'] = $input['variant_name'];
            }
            if (array_key_exists('img', $input)) {
                $payload['img'] = $input['img'];
            }
            if (array_key_exists('catg_ids', $input)) {
                if (is_array($input['catg_ids'])) {
                    $payload['catg_ids'] = json_encode(array_values($input['catg_ids']), JSON_UNESCAPED_UNICODE);
                } else {
                    $payload['catg_ids'] = $input['catg_ids'];
                }
            }
            if (array_key_exists('detail', $input)) {
                if (is_array($input['detail'])) {
                    $payload['detail'] = json_encode($input['detail'], JSON_UNESCAPED_UNICODE);
                } elseif (is_string($input['detail']) && $input['detail'] !== '') {
                    $payload['detail'] = json_encode($input['detail'], JSON_UNESCAPED_UNICODE);
                } else {
                    $payload['detail'] = null;
                }
            }
            if (array_key_exists('status', $input)) {
                $payload['status'] = (int)$input['status'];
            }

            $ok = $productModel->update($id, $payload);
            echo json_encode(['success' => (bool)$ok]);
            break;
        }

        case 'delete': {
            $id = (int)($_POST['id'] ?? 0);
            if ($id <= 0) {
                throw new InvalidArgumentException('ID de producto inválido');
            }
            $ok = $productModel->delete($id);
            echo json_encode(['success' => (bool)$ok]);
            break;
        }

        case 'upload_image': {
            if (empty($_FILES['file']) || !is_uploaded_file($_FILES['file']['tmp_name'])) {
                throw new InvalidArgumentException('No se recibió archivo de imagen.');
            }
            $file = $_FILES['file'];
            $allowed = ['image/jpeg','image/png','image/gif','image/webp'];
            if (!in_array($file['type'], $allowed, true)) {
                throw new InvalidArgumentException('Tipo de imagen no permitido.');
            }

            $ext = pathinfo($file['name'], PATHINFO_EXTENSION);
            $ext = $ext ? ('.' . strtolower($ext)) : '';
            $newName = uniqid('prod_', true) . $ext;

            $baseDir = __DIR__ . '/../../assets/uploads/productos';
            if (!is_dir($baseDir)) {
                mkdir($baseDir, 0775, true);
            }
            $destPath = $baseDir . '/' . $newName;

            if (!move_uploaded_file($file['tmp_name'], $destPath)) {
                throw new RuntimeException('No se pudo guardar la imagen en el servidor.');
            }

            $url = (defined('APP_BASE_PATH') ? APP_BASE_PATH : '/') . 'assets/uploads/productos/' . $newName;
            echo json_encode(['success' => true, 'url' => $url]);
            break;
        }

        default:
            echo json_encode(['success' => false, 'message' => 'Acción no soportada']);
    }
} catch (Throwable $e) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => $e->getMessage()]);
}
