<?php
require_once __DIR__ . '/../../config/auth.php';
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../../models/mautic/MauticSync.php';

requireLogin();
requireModuleAccess('campanas');

header('Content-Type: application/json; charset=utf-8');

function read_input(): array
{
    $raw = file_get_contents('php://input');
    if ($raw === false || $raw === '') {
        return $_POST ?? [];
    }
    $data = json_decode($raw, true);
    return is_array($data) ? $data : ($_POST ?? []);
}

try {
    $db = (new Database())->getConnection();
    $action = $_GET['action'] ?? '';

    switch ($action) {
        case 'list_instances': {
            $stmt = $db->query(
                "SELECT id, nombre, base_url, auth_type, api_user, activo, created_at, updated_at,
                        oauth_client_id, oauth_redirect_uri, oauth_expires_at, min_campaign_id
                 FROM mautic_instancias
                 ORDER BY id DESC"
            );
            echo json_encode(['success' => true, 'data' => $stmt->fetchAll(PDO::FETCH_ASSOC)]);
            break;
        }
        case 'save_instance': {
            $input = read_input();
            $id = isset($input['id']) ? (int)$input['id'] : 0;
            $nombre = trim((string)($input['nombre'] ?? ''));
            $baseUrl = trim((string)($input['base_url'] ?? ''));
            $minCampaignId = isset($input['min_campaign_id']) ? (int)$input['min_campaign_id'] : null;
            $authType = trim((string)($input['auth_type'] ?? 'basic'));
            $apiUser = trim((string)($input['api_user'] ?? ''));
            $apiPass = (string)($input['api_pass'] ?? '');
            $oauthClientId = trim((string)($input['oauth_client_id'] ?? ''));
            $oauthClientSecret = (string)($input['oauth_client_secret'] ?? '');
            $oauthRedirect = trim((string)($input['oauth_redirect_uri'] ?? ''));
            $oauthAccessToken = (string)($input['oauth_access_token'] ?? '');
            $oauthRefreshToken = (string)($input['oauth_refresh_token'] ?? '');
            $activo = !empty($input['activo']) ? 1 : 0;

            if ($baseUrl === '') {
                throw new InvalidArgumentException('Base URL es requerida');
            }

            if ($id > 0) {
                if ($authType === 'basic' && $apiPass === '') {
                    $stmt = $db->prepare("SELECT api_pass FROM mautic_instancias WHERE id = :id");
                    $stmt->execute([':id' => $id]);
                    $apiPass = (string)($stmt->fetchColumn() ?: '');
                }
                if ($authType === 'oauth2' && $oauthClientSecret === '') {
                    $stmt = $db->prepare("SELECT oauth_client_secret FROM mautic_instancias WHERE id = :id");
                    $stmt->execute([':id' => $id]);
                    $oauthClientSecret = (string)($stmt->fetchColumn() ?: '');
                }
                $stmt = $db->prepare(
                    "UPDATE mautic_instancias
                     SET nombre = :nombre,
                         base_url = :base_url,
                         min_campaign_id = :min_campaign_id,
                         auth_type = :auth_type,
                         api_user = :api_user,
                         api_pass = :api_pass,
                         oauth_client_id = :oauth_client_id,
                         oauth_client_secret = :oauth_client_secret,
                         oauth_redirect_uri = :oauth_redirect_uri,
                         oauth_access_token = :oauth_access_token,
                         oauth_refresh_token = :oauth_refresh_token,
                         activo = :activo
                     WHERE id = :id"
                );
                $stmt->execute([
                    ':nombre' => $nombre,
                    ':base_url' => $baseUrl,
                    ':min_campaign_id' => $minCampaignId,
                    ':auth_type' => $authType,
                    ':api_user' => $apiUser,
                    ':api_pass' => $apiPass,
                    ':oauth_client_id' => $oauthClientId,
                    ':oauth_client_secret' => $oauthClientSecret,
                    ':oauth_redirect_uri' => $oauthRedirect,
                    ':oauth_access_token' => $oauthAccessToken,
                    ':oauth_refresh_token' => $oauthRefreshToken,
                    ':activo' => $activo,
                    ':id' => $id,
                ]);
                echo json_encode(['success' => true, 'id' => $id]);
                break;
            }

            if ($authType === 'basic') {
                if ($apiUser === '' || $apiPass === '') {
                    throw new InvalidArgumentException('Usuario y password requeridos');
                }
            }
            if ($authType === 'oauth2') {
                if ($oauthClientId === '' || $oauthClientSecret === '' || $oauthRefreshToken === '') {
                    throw new InvalidArgumentException('Client ID, Client Secret y Refresh Token requeridos');
                }
            }

            $stmt = $db->prepare(
                "INSERT INTO mautic_instancias
                    (nombre, base_url, min_campaign_id, auth_type, api_user, api_pass,
                     oauth_client_id, oauth_client_secret, oauth_redirect_uri,
                     oauth_access_token, oauth_refresh_token, activo)
                 VALUES
                    (:nombre, :base_url, :min_campaign_id, :auth_type, :api_user, :api_pass,
                     :oauth_client_id, :oauth_client_secret, :oauth_redirect_uri,
                     :oauth_access_token, :oauth_refresh_token, :activo)"
            );
            $stmt->execute([
                ':nombre' => $nombre,
                ':base_url' => $baseUrl,
                ':min_campaign_id' => $minCampaignId,
                ':auth_type' => $authType,
                ':api_user' => $apiUser,
                ':api_pass' => $apiPass,
                ':oauth_client_id' => $oauthClientId,
                ':oauth_client_secret' => $oauthClientSecret,
                ':oauth_redirect_uri' => $oauthRedirect,
                ':oauth_access_token' => $oauthAccessToken,
                ':oauth_refresh_token' => $oauthRefreshToken,
                ':activo' => $activo,
            ]);
            echo json_encode(['success' => true, 'id' => (int)$db->lastInsertId()]);
            break;
        }
        case 'test_instance': {
            $input = read_input();
            $baseUrl = trim((string)($input['base_url'] ?? ''));
            $authType = trim((string)($input['auth_type'] ?? 'basic'));
            $apiUser = trim((string)($input['api_user'] ?? ''));
            $apiPass = (string)($input['api_pass'] ?? '');
            $oauthClientId = trim((string)($input['oauth_client_id'] ?? ''));
            $oauthClientSecret = (string)($input['oauth_client_secret'] ?? '');
            $oauthRedirect = trim((string)($input['oauth_redirect_uri'] ?? ''));
            $oauthAccessToken = (string)($input['oauth_access_token'] ?? '');
            $oauthRefreshToken = (string)($input['oauth_refresh_token'] ?? '');

            if ($baseUrl === '') {
                throw new InvalidArgumentException('Base URL es requerida');
            }

            if ($authType === 'basic') {
                if ($apiUser === '' || $apiPass === '') {
                    throw new InvalidArgumentException('Usuario y password requeridos');
                }
                $client = new MauticApiClient($baseUrl, [
                    'auth_type' => 'basic',
                    'api_user' => $apiUser,
                    'api_pass' => $apiPass,
                ]);
                $response = $client->get('users/self');
                $user = $response['user']['username'] ?? null;
                echo json_encode(['success' => true, 'message' => 'Conexion OK', 'user' => $user]);
                break;
            }

            if ($oauthClientId === '' || $oauthClientSecret === '' || $oauthRefreshToken === '') {
                throw new InvalidArgumentException('Client ID, Client Secret y Refresh Token requeridos');
            }

            $tokenUrl = rtrim($baseUrl, '/') . '/oauth/v2/token';
            $payload = [
                'grant_type' => 'refresh_token',
                'client_id' => $oauthClientId,
                'client_secret' => $oauthClientSecret,
                'refresh_token' => $oauthRefreshToken,
                'redirect_uri' => $oauthRedirect,
            ];

            $ch = curl_init($tokenUrl);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($payload));
            curl_setopt($ch, CURLOPT_TIMEOUT, 30);
            $tokenResponse = curl_exec($ch);
            if ($tokenResponse === false) {
                $error = curl_error($ch);
                curl_close($ch);
                throw new RuntimeException('Error OAuth2: ' . $error);
            }
            $tokenCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            curl_close($ch);

            $tokenData = json_decode($tokenResponse, true);
            if ($tokenCode >= 400 || !is_array($tokenData) || empty($tokenData['access_token'])) {
                $message = $tokenData['error_description'] ?? $tokenData['error'] ?? $tokenResponse;
                throw new RuntimeException('OAuth2 refresh error: ' . $message);
            }

            $accessToken = (string)($tokenData['access_token'] ?? $oauthAccessToken);
            $client = new MauticApiClient($baseUrl, [
                'auth_type' => 'oauth2',
                'access_token' => $accessToken,
            ]);
            $response = $client->get('users/self');
            $user = $response['user']['username'] ?? null;
            echo json_encode(['success' => true, 'message' => 'Conexion OK', 'user' => $user]);
            break;
        }
        case 'delete_instance': {
            $input = read_input();
            $id = isset($input['id']) ? (int)$input['id'] : 0;
            if ($id <= 0) {
                throw new InvalidArgumentException('ID invalido');
            }
            $stmt = $db->prepare("DELETE FROM mautic_instancias WHERE id = :id");
            $stmt->execute([':id' => $id]);
            echo json_encode(['success' => true]);
            break;
        }
        case 'list_segments': {
            $instanceId = isset($_GET['instance_id']) ? (int)$_GET['instance_id'] : 0;
            if ($instanceId > 0) {
                $stmt = $db->prepare(
                    "SELECT id, mautic_instancia_id, segmento_clave, mautic_segment_id, descripcion
                     FROM mautic_segmentos_map
                     WHERE mautic_instancia_id = :instancia_id
                     ORDER BY id DESC"
                );
                $stmt->execute([':instancia_id' => $instanceId]);
            } else {
                $stmt = $db->query(
                    "SELECT id, mautic_instancia_id, segmento_clave, mautic_segment_id, descripcion
                     FROM mautic_segmentos_map
                     ORDER BY id DESC"
                );
            }
            echo json_encode(['success' => true, 'data' => $stmt->fetchAll(PDO::FETCH_ASSOC)]);
            break;
        }
        case 'save_segment': {
            $input = read_input();
            $id = isset($input['id']) ? (int)$input['id'] : 0;
            $instanceId = (int)($input['mautic_instancia_id'] ?? 0);
            $clave = trim((string)($input['segmento_clave'] ?? ''));
            $segmentId = (int)($input['mautic_segment_id'] ?? 0);
            $descripcion = trim((string)($input['descripcion'] ?? ''));

            if ($instanceId <= 0 || $clave === '' || $segmentId <= 0) {
                throw new InvalidArgumentException('Instancia, clave y ID son requeridos');
            }

            if ($id > 0) {
                $stmt = $db->prepare(
                    "UPDATE mautic_segmentos_map
                     SET mautic_instancia_id = :instancia_id, segmento_clave = :clave, mautic_segment_id = :segment_id, descripcion = :descripcion
                     WHERE id = :id"
                );
                $stmt->execute([
                    ':instancia_id' => $instanceId,
                    ':clave' => $clave,
                    ':segment_id' => $segmentId,
                    ':descripcion' => $descripcion,
                    ':id' => $id,
                ]);
                echo json_encode(['success' => true, 'id' => $id]);
                break;
            }

            $stmt = $db->prepare(
                "INSERT INTO mautic_segmentos_map (mautic_instancia_id, segmento_clave, mautic_segment_id, descripcion)
                 VALUES (:instancia_id, :clave, :segment_id, :descripcion)"
            );
            $stmt->execute([
                ':instancia_id' => $instanceId,
                ':clave' => $clave,
                ':segment_id' => $segmentId,
                ':descripcion' => $descripcion,
            ]);
            echo json_encode(['success' => true, 'id' => (int)$db->lastInsertId()]);
            break;
        }
        case 'delete_segment': {
            $input = read_input();
            $id = isset($input['id']) ? (int)$input['id'] : 0;
            if ($id <= 0) {
                throw new InvalidArgumentException('ID invalido');
            }
            $stmt = $db->prepare("DELETE FROM mautic_segmentos_map WHERE id = :id");
            $stmt->execute([':id' => $id]);
            echo json_encode(['success' => true]);
            break;
        }
        case 'list_reports': {
            $instanceId = isset($_GET['instance_id']) ? (int)$_GET['instance_id'] : 0;
            if ($instanceId > 0) {
                $stmt = $db->prepare(
                    "SELECT id, mautic_instancia_id, report_key, report_id, campo_campania, campo_lead_id, campo_email,
                            campo_subject, campo_clicks, campo_date_sent, campo_date_read, activo
                     FROM mautic_reportes_map
                     WHERE mautic_instancia_id = :instancia_id
                     ORDER BY id DESC"
                );
                $stmt->execute([':instancia_id' => $instanceId]);
            } else {
                $stmt = $db->query(
                    "SELECT id, mautic_instancia_id, report_key, report_id, campo_campania, campo_lead_id, campo_email,
                            campo_subject, campo_clicks, campo_date_sent, campo_date_read, activo
                     FROM mautic_reportes_map
                     ORDER BY id DESC"
                );
            }
            echo json_encode(['success' => true, 'data' => $stmt->fetchAll(PDO::FETCH_ASSOC)]);
            break;
        }
        case 'save_report': {
            $input = read_input();
            $id = isset($input['id']) ? (int)$input['id'] : 0;
            $instanceId = (int)($input['mautic_instancia_id'] ?? 0);
            $reportKey = trim((string)($input['report_key'] ?? ''));
            $reportId = (int)($input['report_id'] ?? 0);
            $campoCampania = trim((string)($input['campo_campania'] ?? ''));
            $campoLeadId = trim((string)($input['campo_lead_id'] ?? ''));
            $campoEmail = trim((string)($input['campo_email'] ?? ''));
            $campoSubject = trim((string)($input['campo_subject'] ?? ''));
            $campoClicks = trim((string)($input['campo_clicks'] ?? ''));
            $campoDateSent = trim((string)($input['campo_date_sent'] ?? ''));
            $campoDateRead = trim((string)($input['campo_date_read'] ?? ''));
            $activo = !empty($input['activo']) ? 1 : 0;

            if ($instanceId <= 0 || $reportKey === '' || $reportId <= 0 || $campoCampania === '' || $campoEmail === '') {
                throw new InvalidArgumentException('Campos requeridos incompletos');
            }

            if ($id > 0) {
                $stmt = $db->prepare(
                    "UPDATE mautic_reportes_map
                     SET mautic_instancia_id = :instancia_id, report_key = :report_key, report_id = :report_id,
                         campo_campania = :campo_campania, campo_lead_id = :campo_lead_id, campo_email = :campo_email,
                         campo_subject = :campo_subject, campo_clicks = :campo_clicks, campo_date_sent = :campo_date_sent, campo_date_read = :campo_date_read,
                         activo = :activo
                     WHERE id = :id"
                );
                $stmt->execute([
                    ':instancia_id' => $instanceId,
                    ':report_key' => $reportKey,
                    ':report_id' => $reportId,
                    ':campo_campania' => $campoCampania,
                    ':campo_lead_id' => $campoLeadId,
                    ':campo_email' => $campoEmail,
                    ':campo_subject' => $campoSubject,
                    ':campo_clicks' => $campoClicks,
                    ':campo_date_sent' => $campoDateSent,
                    ':campo_date_read' => $campoDateRead,
                    ':activo' => $activo,
                    ':id' => $id,
                ]);
                echo json_encode(['success' => true, 'id' => $id]);
                break;
            }

            $stmt = $db->prepare(
                "INSERT INTO mautic_reportes_map
                    (mautic_instancia_id, report_key, report_id, campo_campania, campo_lead_id, campo_email,
                     campo_subject, campo_clicks, campo_date_sent, campo_date_read, activo)
                 VALUES
                    (:instancia_id, :report_key, :report_id, :campo_campania, :campo_lead_id, :campo_email,
                     :campo_subject, :campo_clicks, :campo_date_sent, :campo_date_read, :activo)"
            );
            $stmt->execute([
                ':instancia_id' => $instanceId,
                ':report_key' => $reportKey,
                ':report_id' => $reportId,
                ':campo_campania' => $campoCampania,
                ':campo_lead_id' => $campoLeadId,
                ':campo_email' => $campoEmail,
                ':campo_subject' => $campoSubject,
                ':campo_clicks' => $campoClicks,
                ':campo_date_sent' => $campoDateSent,
                ':campo_date_read' => $campoDateRead,
                ':activo' => $activo,
            ]);
            echo json_encode(['success' => true, 'id' => (int)$db->lastInsertId()]);
            break;
        }
        case 'delete_report': {
            $input = read_input();
            $id = isset($input['id']) ? (int)$input['id'] : 0;
            if ($id <= 0) {
                throw new InvalidArgumentException('ID invalido');
            }
            $stmt = $db->prepare("DELETE FROM mautic_reportes_map WHERE id = :id");
            $stmt->execute([':id' => $id]);
            echo json_encode(['success' => true]);
            break;
        }
        default:
            http_response_code(404);
            echo json_encode(['success' => false, 'message' => 'Accion no valida']);
    }
} catch (Throwable $e) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => $e->getMessage()]);
}
