<?php
header('Content-Type: application/json; charset=utf-8');
date_default_timezone_set('America/Lima');

require_once __DIR__ . '/../../config/auth.php';
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../../models/EmpresaModel.php';
require_once __DIR__ . '/../../models/ContactoEmpresaModel.php';
require_once __DIR__ . '/../../models/ContactoEmailModel.php';
require_once __DIR__ . '/../../models/ContactoTelefonoModel.php';
require_once __DIR__ . '/../../models/EmpresaHistorialModel.php';
require_once __DIR__ . '/../../models/ContactoHistorialModel.php';

requireLogin();
requireModuleAccess('empresas');

try {
    $db = (new Database())->getConnection();
    $empresa = new EmpresaModel($db);
    $contacto = new ContactoEmpresaModel($db);
    $contactoEmail = new ContactoEmailModel($db);
    $contactoTelefono = new ContactoTelefonoModel($db);
    $empresaHistorial = new EmpresaHistorialModel($db);
    $contactoHistorial = new ContactoHistorialModel($db);
} catch (Throwable $e) {
    http_response_code(500);
    echo json_encode(['success' => false, 'message' => 'Error de conexion.']);
    exit();
}

$action = $_REQUEST['action'] ?? 'list';

try {
    switch ($action) {
        case 'list': {
            $term = trim((string)($_GET['q'] ?? ''));
            $by = trim((string)($_GET['by'] ?? 'todos'));
            $page = max(1, (int)($_GET['page'] ?? 1));
            $limit = max(1, (int)($_GET['limit'] ?? 25));
            $offset = ($page - 1) * $limit;
            $filters = ['q' => $term, 'by' => $by];
            $rows = $empresa->list($filters, $limit, $offset);
            $total = $empresa->count($filters);
            echo json_encode(['success' => true, 'data' => $rows, 'total' => $total]);
            break;
        }

        case 'get': {
            $id = (int)($_GET['id'] ?? 0);
            if ($id <= 0) {
                throw new InvalidArgumentException('ID invalido');
            }
            $row = $empresa->get($id);
            if (!$row) {
                echo json_encode(['success' => false, 'message' => 'No encontrado']);
                break;
            }
            $historial = $empresaHistorial->listByEmpresa($id);
            echo json_encode(['success' => true, 'data' => $row, 'historial' => $historial]);
            break;
        }

        case 'create': {
            $input = json_decode(file_get_contents('php://input'), true);
            if (!is_array($input)) {
                $input = $_POST;
            }
            $ruc = trim((string)($input['ruc'] ?? ''));
            $razon = trim((string)($input['razon_social'] ?? ''));
            if ($ruc === '' || $razon === '') {
                echo json_encode(['success' => false, 'message' => 'RUC y razon social son obligatorios.']);
                break;
            }
            $id = $empresa->create($input);
            $usuarioId = (int)($_SESSION['idusuario'] ?? 0);
            if ($usuarioId > 0) {
                $empresaHistorial->add($id, $usuarioId, 'creacion');
            }
            echo json_encode(['success' => true, 'id' => $id, 'message' => 'Empresa creada correctamente.']);
            break;
        }

        case 'update': {
            $input = json_decode(file_get_contents('php://input'), true);
            if (!is_array($input)) {
                $input = $_POST;
            }
            $id = (int)($input['id'] ?? 0);
            if ($id <= 0) {
                throw new InvalidArgumentException('ID invalido');
            }
            $ok = $empresa->update($id, $input);
            $usuarioId = (int)($_SESSION['idusuario'] ?? 0);
            if ($usuarioId > 0) {
                $empresaHistorial->add($id, $usuarioId, 'actualizacion');
            }
            echo json_encode(['success' => $ok, 'message' => $ok ? 'Empresa actualizada.' : 'No se pudo actualizar.']);
            break;
        }

        case 'delete': {
            $input = json_decode(file_get_contents('php://input'), true);
            if (!is_array($input)) {
                $input = $_POST;
            }
            $id = (int)($input['id'] ?? 0);
            if ($id <= 0) {
                throw new InvalidArgumentException('ID invalido');
            }
            $ok = $empresa->delete($id);
            echo json_encode(['success' => $ok, 'message' => $ok ? 'Empresa eliminada.' : 'No se pudo eliminar.']);
            break;
        }

        case 'list_contactos': {
            $empresaId = (int)($_GET['empresa_id'] ?? 0);
            if ($empresaId <= 0) {
                throw new InvalidArgumentException('Empresa invalida');
            }
            $rows = $contacto->listByEmpresa($empresaId);
            echo json_encode(['success' => true, 'data' => $rows]);
            break;
        }

        case 'get_contacto': {
            $id = (int)($_GET['id'] ?? 0);
            if ($id <= 0) {
                throw new InvalidArgumentException('ID invalido');
            }
            $row = $contacto->get($id);
            if (!$row) {
                echo json_encode(['success' => false, 'message' => 'No encontrado']);
                break;
            }
            $telefonos = $contactoTelefono->listByContacto($id);
            $emails = $contactoEmail->listByContacto($id);
            foreach ($telefonos as $t) {
                if (!empty($t['es_principal'])) {
                    $row['celular'] = $t['telefono'];
                    break;
                }
            }
            foreach ($emails as $e) {
                if (!empty($e['es_principal'])) {
                    $row['email'] = $e['email'];
                    break;
                }
            }
            echo json_encode(['success' => true, 'data' => $row]);
            break;
        }

        case 'create_contacto': {
            $input = json_decode(file_get_contents('php://input'), true);
            if (!is_array($input)) {
                $input = $_POST;
            }
            $empresaId = (int)($input['empresa_id'] ?? 0);
            if ($empresaId <= 0) {
                throw new InvalidArgumentException('Empresa invalida');
            }
            $nombres = trim((string)($input['nombres'] ?? ''));
            if ($nombres === '') {
                echo json_encode(['success' => false, 'message' => 'Nombres es obligatorio.']);
                break;
            }
            $origen = trim((string)($input['origen'] ?? ''));
            if ($origen === '') {
                echo json_encode(['success' => false, 'message' => 'Origen es obligatorio.']);
                break;
            }
            $celular = preg_replace('/\D+/', '', (string)($input['celular'] ?? ''));
            $email = strtolower(trim((string)($input['email'] ?? '')));
            if ($celular !== '') {
                $stmt = $db->prepare(
                    "SELECT 1 FROM contacto_telefonos WHERE telefono = :telefono AND estado = 1 LIMIT 1"
                );
                $stmt->execute([':telefono' => $celular]);
                if ($stmt->fetchColumn()) {
                    echo json_encode(['success' => false, 'message' => 'El celular ya existe.']);
                    break;
                }
            }
            if ($email !== '') {
                $stmt = $db->prepare(
                    "SELECT 1 FROM contacto_emails WHERE email = :email AND estado = 1 LIMIT 1"
                );
                $stmt->execute([':email' => $email]);
                if ($stmt->fetchColumn()) {
                    echo json_encode(['success' => false, 'message' => 'El email ya existe.']);
                    break;
                }
            }
            $id = $contacto->create($empresaId, $input);
            $telefonos = [];
            if ($celular !== '') {
                $telefonos[] = ['telefono' => $celular, 'tipo' => 'movil', 'es_principal' => 1];
            }
            $emails = [];
            if ($email !== '') {
                $emails[] = ['email' => $email, 'tipo' => 'trabajo', 'es_principal' => 1];
            }
            $contactoTelefono->sync($id, $telefonos);
            $contactoEmail->sync($id, $emails);
            $usuarioId = (int)($_SESSION['idusuario'] ?? 0);
            if ($usuarioId > 0) {
                $contactoHistorial->add($id, $usuarioId, 'creacion');
            }
            echo json_encode(['success' => true, 'id' => $id, 'message' => 'Contacto creado correctamente.']);
            break;
        }

        case 'update_contacto': {
            $input = json_decode(file_get_contents('php://input'), true);
            if (!is_array($input)) {
                $input = $_POST;
            }
            $id = (int)($input['id'] ?? 0);
            if ($id <= 0) {
                throw new InvalidArgumentException('ID invalido');
            }
            $ok = $contacto->update($id, $input);
            $celular = preg_replace('/\D+/', '', (string)($input['celular'] ?? ''));
            $email = strtolower(trim((string)($input['email'] ?? '')));
            $telefonos = [];
            if ($celular !== '') {
                $telefonos[] = ['telefono' => $celular, 'tipo' => 'movil', 'es_principal' => 1];
            }
            $emails = [];
            if ($email !== '') {
                $emails[] = ['email' => $email, 'tipo' => 'trabajo', 'es_principal' => 1];
            }
            $contactoTelefono->sync($id, $telefonos);
            $contactoEmail->sync($id, $emails);
            $usuarioId = (int)($_SESSION['idusuario'] ?? 0);
            if ($usuarioId > 0) {
                $contactoHistorial->add($id, $usuarioId, 'actualizacion');
            }
            echo json_encode(['success' => $ok, 'message' => $ok ? 'Contacto actualizado.' : 'No se pudo actualizar.']);
            break;
        }

        case 'delete_contacto': {
            $input = json_decode(file_get_contents('php://input'), true);
            if (!is_array($input)) {
                $input = $_POST;
            }
            $id = (int)($input['id'] ?? 0);
            if ($id <= 0) {
                throw new InvalidArgumentException('ID invalido');
            }
            $ok = $contacto->delete($id);
            echo json_encode(['success' => $ok, 'message' => $ok ? 'Contacto eliminado.' : 'No se pudo eliminar.']);
            break;
        }

        default:
            echo json_encode(['success' => false, 'message' => 'Accion no soportada.']);
            break;
    }
} catch (Throwable $e) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'No se pudo procesar la solicitud.']);
}
