<?php
header('Content-Type: application/json; charset=utf-8');
date_default_timezone_set('America/Lima');

require_once __DIR__ . '/../../config/auth.php';
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../../models/ContactoModel.php';
require_once __DIR__ . '/../../models/ContactoEmailModel.php';
require_once __DIR__ . '/../../models/ContactoTelefonoModel.php';
require_once __DIR__ . '/../../models/ContactoHistorialModel.php';

requireLogin();
requireModuleAccess('contactos_empresa');

try {
    $db = (new Database())->getConnection();
    $contacto = new ContactoModel($db);
    $contactoEmail = new ContactoEmailModel($db);
    $contactoTelefono = new ContactoTelefonoModel($db);
    $contactoHistorial = new ContactoHistorialModel($db);
} catch (Throwable $e) {
    http_response_code(500);
    echo json_encode(['success' => false, 'message' => 'Error de conexion.']);
    exit();
}

$action = $_REQUEST['action'] ?? 'list';

function normalize_phone(string $value): string
{
    return preg_replace('/\D+/', '', $value);
}

function normalize_email(string $value): string
{
    return strtolower(trim($value));
}

function is_valid_phone(string $value): bool
{
    if ($value === '') {
        return true;
    }
    $digits = normalize_phone($value);
    $len = strlen($digits);
    return $len >= 6 && $len <= 15;
}

function is_valid_email(string $value): bool
{
    if ($value === '') {
        return true;
    }
    return (bool)filter_var($value, FILTER_VALIDATE_EMAIL);
}

function findDuplicateValues(PDO $db, int $contactoId, array $telefonos, array $emails): ?string
{
    if ($telefonos) {
        $placeholders = implode(',', array_fill(0, count($telefonos), '?'));
        $sql = "SELECT ct.contacto_id, ct.telefono
                FROM contacto_telefonos ct
                WHERE ct.contacto_id <> ? AND ct.estado = 1 AND ct.telefono IN ($placeholders)
                LIMIT 1";
        $stmt = $db->prepare($sql);
        $stmt->execute(array_merge([$contactoId], $telefonos));
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        if ($row) {
            return 'El telefono ya esta registrado en otro contacto.';
        }
    }

    if ($emails) {
        $placeholders = implode(',', array_fill(0, count($emails), '?'));
        $sql = "SELECT ce.contacto_id, ce.email
                FROM contacto_emails ce
                WHERE ce.contacto_id <> ? AND ce.estado = 1 AND ce.email IN ($placeholders)
                LIMIT 1";
        $stmt = $db->prepare($sql);
        $stmt->execute(array_merge([$contactoId], $emails));
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        if ($row) {
            return 'El correo ya esta registrado en otro contacto.';
        }
    }

    return null;
}

try {
    switch ($action) {
        case 'list': {
            $q = $_GET['q'] ?? '';
            $campo = $_GET['campo'] ?? 'todos';
            $page = max(1, (int)($_GET['page'] ?? 1));
            $limit = max(1, (int)($_GET['limit'] ?? 25));
            $offset = ($page - 1) * $limit;
            $filters = [];
            if ($q !== '') $filters['q'] = $q;
            if ($campo !== '') $filters['campo'] = $campo;
            $rows = $contacto->list($filters, $limit, $offset);
            $total = $contacto->count($filters);
            echo json_encode(['success' => true, 'data' => $rows, 'total' => $total]);
            break;
        }

        case 'get': {
            $id = (int)($_GET['id'] ?? 0);
            if ($id <= 0) throw new InvalidArgumentException('ID invalido');
            $row = $contacto->get($id);
            if (!$row) { echo json_encode(['success' => false, 'message' => 'No encontrado']); break; }
            $telefonosRows = $contactoTelefono->listByContacto($id);
            $emailsRows = $contactoEmail->listByContacto($id);
            $historial = $contactoHistorial->listByContacto($id);
            $telefonos = [];
            $emails = [];
            $celular = '';
            $telefonoFijo = '';
            foreach ($telefonosRows as $t) {
                $isPrincipal = !empty($t['es_principal']);
                $tipo = $t['tipo'] ?? 'otro';
                if ($isPrincipal && $celular === '') {
                    $celular = $t['telefono'];
                    continue;
                }
                if ($tipo === 'fijo' && $telefonoFijo === '') {
                    $telefonoFijo = $t['telefono'];
                    continue;
                }
                $telefonos[] = ['telefono' => $t['telefono'], 'tipo' => $tipo];
            }
            $emailPrincipal = '';
            $emailTipo = 'trabajo';
            foreach ($emailsRows as $e) {
                $isPrincipal = !empty($e['es_principal']);
                $tipo = $e['tipo'] ?? 'otro';
                if ($isPrincipal && $emailPrincipal === '') {
                    $emailPrincipal = $e['email'];
                    $emailTipo = $tipo ?: 'trabajo';
                    continue;
                }
                $emails[] = ['email' => $e['email'], 'tipo' => $tipo];
            }
            $row['celular'] = $celular;
            $row['telefono_fijo'] = $telefonoFijo;
            $row['telefonos'] = $telefonos;
            $row['email'] = $emailPrincipal;
            $row['email_tipo'] = $emailTipo;
            $row['emails'] = $emails;
            echo json_encode(['success' => true, 'data' => $row, 'historial' => $historial]);
            break;
        }

        case 'create': {
            $input = json_decode(file_get_contents('php://input'), true);
            if (!is_array($input)) $input = $_POST;

            $empresaId = (int)($input['empresa_id'] ?? 0);
            if ($empresaId <= 0) throw new InvalidArgumentException('Empresa invalida');
            if (empty($input['nombres'])) throw new InvalidArgumentException('Nombres es requerido');

            $celular = trim((string)($input['celular'] ?? ''));
            $telefonoFijo = trim((string)($input['telefono_fijo'] ?? ''));
            $email = trim((string)($input['email'] ?? ''));
            $emailTipo = strtolower(trim((string)($input['email_tipo'] ?? 'trabajo')));
            if (!is_valid_phone($celular)) throw new InvalidArgumentException('Celular invalido');
            if (!is_valid_phone($telefonoFijo)) throw new InvalidArgumentException('Telefono fijo invalido');
            if (!is_valid_email($email)) throw new InvalidArgumentException('Email invalido');

            $telefonos = [];
            $celularNorm = normalize_phone($celular);
            if ($celularNorm !== '') {
                $telefonos[] = ['telefono' => $celularNorm, 'tipo' => 'movil', 'es_principal' => 1];
            }
            $telefonoFijoNorm = normalize_phone($telefonoFijo);
            if ($telefonoFijoNorm !== '') {
                $telefonos[] = ['telefono' => $telefonoFijoNorm, 'tipo' => 'fijo', 'es_principal' => 0];
            }
            foreach ((array)($input['telefonos'] ?? []) as $item) {
                if (is_string($item)) {
                    $telefono = normalize_phone($item);
                    $tipo = 'otro';
                } else {
                    $telefono = normalize_phone((string)($item['telefono'] ?? ''));
                    $tipo = strtolower(trim((string)($item['tipo'] ?? 'otro')));
                }
                if ($telefono === '') {
                    continue;
                }
                $telefonos[] = ['telefono' => $telefono, 'tipo' => $tipo, 'es_principal' => 0];
            }

            $emails = [];
            $emailNorm = normalize_email($email);
            if ($emailNorm !== '') {
                $principal = ($emailTipo === 'trabajo') ? 1 : 0;
                $emails[] = ['email' => $emailNorm, 'tipo' => $emailTipo ?: 'trabajo', 'es_principal' => $principal];
            }
            foreach ((array)($input['emails'] ?? []) as $item) {
                if (is_string($item)) {
                    $value = normalize_email($item);
                    $tipo = 'otro';
                } else {
                    $value = normalize_email((string)($item['email'] ?? ''));
                    $tipo = strtolower(trim((string)($item['tipo'] ?? 'otro')));
                }
                if ($value === '') {
                    continue;
                }
                $emails[] = ['email' => $value, 'tipo' => $tipo, 'es_principal' => 0];
            }

            $telefonosFlat = array_values(array_unique(array_map(static fn($t) => $t['telefono'], $telefonos)));
            $emailsFlat = array_values(array_unique(array_map(static fn($e) => $e['email'], $emails)));
            $dupMsg = findDuplicateValues($db, 0, $telefonosFlat, $emailsFlat);
            if ($dupMsg) {
                echo json_encode(['success' => false, 'message' => $dupMsg]);
                break;
            }
            foreach ($telefonosFlat as $t) {
                if (!is_valid_phone($t)) throw new InvalidArgumentException('Telefono adicional invalido');
            }
            foreach ($emailsFlat as $em) {
                if (!is_valid_email($em)) throw new InvalidArgumentException('Email adicional invalido');
            }

            $db->beginTransaction();
            $cid = $contacto->create($input);
            $contactoTelefono->sync($cid, $telefonos);
            $contactoEmail->sync($cid, $emails);
            $usuarioId = (int)($_SESSION['idusuario'] ?? 0);
            if ($usuarioId > 0) {
                $contactoHistorial->add($cid, $usuarioId, 'creacion');
            }
            $db->commit();
            echo json_encode(['success' => true, 'id' => $cid]);
            break;
        }

        case 'update': {
            $input = json_decode(file_get_contents('php://input'), true);
            if (!is_array($input)) $input = $_POST;
            $id = (int)($input['id'] ?? 0);
            if ($id <= 0) throw new InvalidArgumentException('ID invalido');

            $celular = trim((string)($input['celular'] ?? ''));
            $telefonoFijo = trim((string)($input['telefono_fijo'] ?? ''));
            $email = trim((string)($input['email'] ?? ''));
            $emailTipo = strtolower(trim((string)($input['email_tipo'] ?? 'trabajo')));
            if (!is_valid_phone($celular)) throw new InvalidArgumentException('Celular invalido');
            if (!is_valid_phone($telefonoFijo)) throw new InvalidArgumentException('Telefono fijo invalido');
            if (!is_valid_email($email)) throw new InvalidArgumentException('Email invalido');

            $telefonos = [];
            $celularNorm = normalize_phone($celular);
            if ($celularNorm !== '') {
                $telefonos[] = ['telefono' => $celularNorm, 'tipo' => 'movil', 'es_principal' => 1];
            }
            $telefonoFijoNorm = normalize_phone($telefonoFijo);
            if ($telefonoFijoNorm !== '') {
                $telefonos[] = ['telefono' => $telefonoFijoNorm, 'tipo' => 'fijo', 'es_principal' => 0];
            }
            foreach ((array)($input['telefonos'] ?? []) as $item) {
                if (is_string($item)) {
                    $telefono = normalize_phone($item);
                    $tipo = 'otro';
                } else {
                    $telefono = normalize_phone((string)($item['telefono'] ?? ''));
                    $tipo = strtolower(trim((string)($item['tipo'] ?? 'otro')));
                }
                if ($telefono === '') {
                    continue;
                }
                $telefonos[] = ['telefono' => $telefono, 'tipo' => $tipo, 'es_principal' => 0];
            }

            $emails = [];
            $emailNorm = normalize_email($email);
            if ($emailNorm !== '') {
                $principal = ($emailTipo === 'trabajo') ? 1 : 0;
                $emails[] = ['email' => $emailNorm, 'tipo' => $emailTipo ?: 'trabajo', 'es_principal' => $principal];
            }
            foreach ((array)($input['emails'] ?? []) as $item) {
                if (is_string($item)) {
                    $value = normalize_email($item);
                    $tipo = 'otro';
                } else {
                    $value = normalize_email((string)($item['email'] ?? ''));
                    $tipo = strtolower(trim((string)($item['tipo'] ?? 'otro')));
                }
                if ($value === '') {
                    continue;
                }
                $emails[] = ['email' => $value, 'tipo' => $tipo, 'es_principal' => 0];
            }

            $telefonosFlat = array_values(array_unique(array_map(static fn($t) => $t['telefono'], $telefonos)));
            $emailsFlat = array_values(array_unique(array_map(static fn($e) => $e['email'], $emails)));
            $dupMsg = findDuplicateValues($db, $id, $telefonosFlat, $emailsFlat);
            if ($dupMsg) {
                echo json_encode(['success' => false, 'message' => $dupMsg]);
                break;
            }
            foreach ($telefonosFlat as $t) {
                if (!is_valid_phone($t)) throw new InvalidArgumentException('Telefono adicional invalido');
            }
            foreach ($emailsFlat as $em) {
                if (!is_valid_email($em)) throw new InvalidArgumentException('Email adicional invalido');
            }

            $db->beginTransaction();
            $ok = $contacto->update($id, $input);
            $contactoTelefono->sync($id, $telefonos);
            $contactoEmail->sync($id, $emails);
            $usuarioId = (int)($_SESSION['idusuario'] ?? 0);
            if ($usuarioId > 0) {
                $contactoHistorial->add($id, $usuarioId, 'actualizacion');
            }
            $db->commit();
            echo json_encode(['success' => (bool)$ok]);
            break;
        }

        case 'delete': {
            $id = (int)($_POST['id'] ?? 0);
            if ($id <= 0) throw new InvalidArgumentException('ID invalido');
            $ok = $contacto->delete($id);
            echo json_encode(['success' => (bool)$ok]);
            break;
        }

        case 'empresa_options': {
            $q = trim((string)($_GET['q'] ?? ''));
            $rows = $contacto->listEmpresas($q);
            echo json_encode(['success' => true, 'data' => $rows]);
            break;
        }

        default:
            echo json_encode(['success' => false, 'message' => 'Accion no soportada']);
    }
} catch (Throwable $e) {
    if ($db->inTransaction()) {
        $db->rollBack();
    }
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'No se pudo procesar la solicitud.']);
}
