<?php
require_once __DIR__ . '/../../config/auth.php';
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../../models/Campania.php';

requireLogin();
requireModuleAccess('campanas');

header('Content-Type: application/json; charset=utf-8');

function campania_input(): array
{
    $raw = file_get_contents('php://input');
    if ($raw === false || $raw === '') {
        return $_POST ?? [];
    }
    $data = json_decode($raw, true);
    if (!is_array($data)) {
        return $_POST ?? [];
    }
    return $data;
}

function build_tree(array $rows): array
{
    $byParent = [];
    foreach ($rows as $row) {
        $parentKey = $row['parent_id'] === null ? 0 : (int)$row['parent_id'];
        if (!isset($byParent[$parentKey])) {
            $byParent[$parentKey] = [];
        }
        $byParent[$parentKey][] = $row;
    }

    $build = function ($parentId) use (&$build, $byParent) {
        $items = $byParent[$parentId] ?? [];
        foreach ($items as &$item) {
            $itemId = (int)$item['id_campania'];
            $item['children'] = $build($itemId);
        }
        return $items;
    };

    return $build(0);
}

try {
    $db = (new Database())->getConnection();
    $model = new Campania($db);

    $action = $_GET['action'] ?? 'list';
    switch ($action) {
        case 'list': {
            $rows = $model->listarTodas();
            echo json_encode(['success' => true, 'data' => build_tree($rows)]);
            break;
        }
        case 'metrics': {
            $id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
            if ($id <= 0) {
                throw new InvalidArgumentException('ID invalido');
            }

            $stmt = $db->prepare(
                "SELECT COUNT(*) AS total_contactos
                 FROM campania_contacto
                 WHERE campania_id = :campania_id"
            );
            $stmt->execute([':campania_id' => $id]);
            $total = (int)($stmt->fetchColumn() ?: 0);

            $stmt = $db->prepare(
                "SELECT
                    COALESCE(SUM(CASE WHEN open = 'Yes' THEN 1 ELSE 0 END), 0) AS open_total,
                    COALESCE(SUM(CASE WHEN clicks > 0 THEN 1 ELSE 0 END), 0) AS clickers_total
                 FROM campania_contacto_eventos
                 WHERE campania_id = :campania_id"
            );
            $stmt->execute([':campania_id' => $id]);
            $row = $stmt->fetch(PDO::FETCH_ASSOC) ?: ['open_total' => 0, 'clickers_total' => 0];

            echo json_encode([
                'success' => true,
                'data' => [
                    'total_contactos' => $total,
                    'open_total' => (int)$row['open_total'],
                    'clickers_total' => (int)$row['clickers_total'],
                ],
            ]);
            break;
        }
        case 'create': {
            $input = campania_input();
            if (empty($input['titulo'])) {
                throw new InvalidArgumentException('Titulo requerido');
            }
            $payload = [
                'parent_id' => $input['parent_id'] ?? null,
                'titulo' => trim((string)$input['titulo']),
                'asunto' => $input['asunto'] ?? null,
                'fecha_inicio' => $input['fecha_inicio'] ?? null,
                'fecha_fin' => $input['fecha_fin'] ?? null,
            ];
            $ok = $model->crear($payload);
            echo json_encode(['success' => (bool)$ok]);
            break;
        }
        case 'update': {
            $input = campania_input();
            $id = isset($input['id']) ? (int)$input['id'] : 0;
            if ($id <= 0) {
                throw new InvalidArgumentException('ID invalido');
            }
            if (empty($input['titulo'])) {
                throw new InvalidArgumentException('Titulo requerido');
            }
            $payload = [
                'titulo' => trim((string)$input['titulo']),
                'asunto' => $input['asunto'] ?? null,
                'fecha_inicio' => $input['fecha_inicio'] ?? null,
                'fecha_fin' => $input['fecha_fin'] ?? null,
            ];
            $ok = $model->actualizar($id, $payload);
            echo json_encode(['success' => (bool)$ok]);
            break;
        }
        case 'delete': {
            $input = campania_input();
            $id = isset($input['id']) ? (int)$input['id'] : 0;
            if ($id <= 0) {
                throw new InvalidArgumentException('ID invalido');
            }
            $ok = $model->eliminar($id);
            echo json_encode(['success' => (bool)$ok]);
            break;
        }
        default:
            http_response_code(404);
            echo json_encode(['success' => false, 'message' => 'Accion no valida']);
    }
} catch (Throwable $e) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => $e->getMessage()]);
}
