<?php
require_once __DIR__ . '/config.php';

/**
 * Obtiene configuración de Telegram desde variables de entorno.
 * Retorna null si no hay token o chat_ids definidos.
 */
function getTelegramConfig(): ?array {
    static $cache = null;
    if ($cache !== null) {
        return $cache;
    }

    $token = $_ENV['TELEGRAM_BOT_TOKEN'] ?? '';
    $chatIdsRaw = $_ENV['TELEGRAM_CHAT_IDS'] ?? '';
    $chatIds = array_values(array_filter(array_map('trim', explode(',', $chatIdsRaw))));

    if ($token === '' || empty($chatIds)) {
        $cache = null;
        return $cache;
    }

    $cache = [
        'token' => $token,
        'chat_ids' => $chatIds,
    ];

    return $cache;
}

/**
 * Envía un mensaje a Telegram. Devuelve true si al menos un envío se
 * intentó con configuración válida; no lanza excepciones para no
 * interrumpir el flujo principal.
 */
function sendTelegramEvent(string $title, array $fields = []): bool {
    $config = getTelegramConfig();
    if (!$config) {
        return false;
    }

    $lines = ["*{$title}*"];
    foreach ($fields as $label => $value) {
        if ($value === null || $value === '') {
            continue;
        }
        $safeValue = is_scalar($value) ? (string) $value : json_encode($value);
        $lines[] = "{$label}: `{$safeValue}`";
    }

    $payloadText = implode("\n", $lines);

    foreach ($config['chat_ids'] as $chatId) {
        $url = "https://api.telegram.org/bot{$config['token']}/sendMessage";
        $postFields = [
            'chat_id' => $chatId,
            'text' => $payloadText,
            'parse_mode' => 'Markdown',
        ];

        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $postFields);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 2);
        curl_exec($ch);
        curl_close($ch);
    }

    return true;
}

/**
 * Enmascara usuario enviando solo dos primeras letras y asteriscos.
 */
function telegramMaskUser(string $user): string {
    $trimmed = trim($user);
    if ($trimmed === '') {
        return 'n/a';
    }
    $prefix = substr($trimmed, 0, 2);
    $stars = str_repeat('*', max(3, strlen($trimmed) - 2));
    return $prefix . $stars;
}
