<?php
require_once __DIR__ . '/database.php';

define('REMEMBER_COOKIE_NAME', 'remember_token');
define('REMEMBER_LIFETIME_DAYS', 30);

function rememberTableEnsure(PDO $db): void {
    $db->exec(
        "CREATE TABLE IF NOT EXISTS remember_tokens (
            id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
            user_id BIGINT UNSIGNED NOT NULL,
            selector CHAR(24) NOT NULL UNIQUE,
            token_hash CHAR(64) NOT NULL,
            user_agent_hash CHAR(64) DEFAULT NULL,
            expires_at DATETIME NOT NULL,
            created_at DATETIME NOT NULL,
            INDEX (user_id),
            INDEX (expires_at)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4"
    );
}

function rememberSetCookie(string $value): void {
    $path = defined('APP_BASE_PATH') ? APP_BASE_PATH : '/';
    setcookie(
        REMEMBER_COOKIE_NAME,
        $value,
        [
            'expires' => time() + (REMEMBER_LIFETIME_DAYS * 24 * 60 * 60),
            'path' => $path,
            'httponly' => true,
            'secure' => !empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off',
            'samesite' => 'Lax',
        ]
    );
}

function rememberClearCookie(): void {
    $path = defined('APP_BASE_PATH') ? APP_BASE_PATH : '/';
    setcookie(
        REMEMBER_COOKIE_NAME,
        '',
        [
            'expires' => time() - 3600,
            'path' => $path,
            'httponly' => true,
            'secure' => !empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off',
            'samesite' => 'Lax',
        ]
    );
}

function rememberIssueToken(PDO $db, int $userId): string {
    rememberTableEnsure($db);

    $selector = base64_encode(random_bytes(9)); // 12 chars
    $validator = base64_encode(random_bytes(32));
    $tokenHash = hash('sha256', $validator);
    $uaHash = hash('sha256', $_SERVER['HTTP_USER_AGENT'] ?? '');

    $expiresAt = (new DateTimeImmutable('now'))->modify('+' . REMEMBER_LIFETIME_DAYS . ' days')->format('Y-m-d H:i:s');
    $now = (new DateTimeImmutable('now'))->format('Y-m-d H:i:s');

    // limpiar tokens previos del usuario
    $stmt = $db->prepare("DELETE FROM remember_tokens WHERE user_id = :uid");
    $stmt->execute([':uid' => $userId]);

    $stmt = $db->prepare("INSERT INTO remember_tokens (user_id, selector, token_hash, user_agent_hash, expires_at, created_at) VALUES (:uid, :selector, :token_hash, :ua, :expires, :created)");
    $stmt->execute([
        ':uid' => $userId,
        ':selector' => $selector,
        ':token_hash' => $tokenHash,
        ':ua' => $uaHash,
        ':expires' => $expiresAt,
        ':created' => $now,
    ]);

    return $selector . ':' . $validator;
}

function rememberDeleteBySelector(PDO $db, string $selector): void {
    $stmt = $db->prepare("DELETE FROM remember_tokens WHERE selector = :selector");
    $stmt->execute([':selector' => $selector]);
}

function rememberTryLogin(): bool {
    if (isLoggedIn()) {
        return true;
    }
    $cookie = $_COOKIE[REMEMBER_COOKIE_NAME] ?? '';
    if (!$cookie || !str_contains($cookie, ':')) {
        return false;
    }

    [$selector, $validator] = explode(':', $cookie, 2);
    if ($selector === '' || $validator === '') {
        rememberClearCookie();
        return false;
    }

    try {
        $db = (new Database())->getConnection();
    } catch (Throwable $e) {
        return false;
    }

    rememberTableEnsure($db);

    $stmt = $db->prepare("SELECT user_id, token_hash, user_agent_hash, expires_at FROM remember_tokens WHERE selector = :selector LIMIT 1");
    $stmt->execute([':selector' => $selector]);
    $row = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$row) {
        rememberClearCookie();
        return false;
    }

    if (new DateTimeImmutable($row['expires_at']) < new DateTimeImmutable('now')) {
        rememberDeleteBySelector($db, $selector);
        rememberClearCookie();
        return false;
    }

    $expected = $row['token_hash'];
    $provided = hash('sha256', $validator);
    $uaHash = hash('sha256', $_SERVER['HTTP_USER_AGENT'] ?? '');

    if (!hash_equals($expected, $provided) || ($row['user_agent_hash'] && !hash_equals($row['user_agent_hash'], $uaHash))) {
        rememberDeleteBySelector($db, $selector);
        rememberClearCookie();
        return false;
    }

    // Cargar datos de usuario
    $stmt = $db->prepare("SELECT idusuario, usuario, area, tipoempleado FROM usuarios WHERE idusuario = :uid LIMIT 1");
    $stmt->execute([':uid' => $row['user_id']]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);
    if (!$user) {
        rememberDeleteBySelector($db, $selector);
        rememberClearCookie();
        return false;
    }

    session_regenerate_id(true);
    $_SESSION['usuario'] = $user['usuario'];
    $_SESSION['area'] = $user['area'];
    $_SESSION['idusuario'] = (int)$user['idusuario'];
    $_SESSION['tipoempleado'] = $user['tipoempleado'] ?? null;

    // Rotar token
    $newValue = rememberIssueToken($db, (int)$user['idusuario']);
    rememberSetCookie($newValue);

    return true;
}
