import { paginacion, updateRowsPerPage } from "../requerimientos/paginacion.js";

const apiUrl = `${base}controller/requerimiento_v2/requerimiento.php`;

const state = { total: 0, editingId: null };

const tableBody = document.getElementById("tabla-requerimientos");
const searchInput = document.getElementById("search-term");
const btnRefresh = document.getElementById("btn-refresh");
const pageSizeSel = document.getElementById("req-pageSize");
const totalLabel = document.getElementById("req-total");
const btnOpenModal = document.getElementById("btn-open-modal");

const formReq = document.getElementById("form-requerimiento");
const btnSave = document.getElementById("btn-save");
const modal = document.getElementById("modal-req");
const btnCloseModal = document.getElementById("btn-close-modal");
const modalTitle = document.getElementById("modal-title");
const empresaInput = document.getElementById("req-empresa-input");
const empresaIdField = document.getElementById("req-empresa-id");
const empresaDropdown = document.getElementById("req-empresa-dropdown");
const contactoSelect = document.getElementById("req-contacto");
const usuarioApoyoSelect = document.getElementById("req-usuario-apoyo");
const btnEmpresaNuevo = document.getElementById("btn-empresa-nuevo");
const empresaQuickModal = document.getElementById("empresaQuickModal");
const btnEmpresaQuickClose = document.getElementById("btnEmpresaQuickClose");
const btnEmpresaQuickCancel = document.getElementById("btnEmpresaQuickCancel");
const btnEmpresaQuickSave = document.getElementById("btnEmpresaQuickSave");
const empresaRuc = document.getElementById("qe_ruc");
const empresaRazon = document.getElementById("qe_razon_social");
const empresaNombre = document.getElementById("qe_nombre_comercial");
const empresaOrigen = document.getElementById("qe_origen");
const btnContactoNuevo = document.getElementById("btn-contacto-nuevo");
const contactoQuickModal = document.getElementById("contactoQuickModal");
const contactoQuickTitle = document.getElementById("contactoQuickTitle");
const btnContactoQuickClose = document.getElementById("btnContactoQuickClose");
const btnContactoQuickCancel = document.getElementById("btnContactoQuickCancel");
const btnContactoQuickSave = document.getElementById("btnContactoQuickSave");
const contactoEmpresaNombre = document.getElementById("contacto-empresa-nombre");
const contactoEmpresaId = document.getElementById("contacto-empresa-id");
const contactoId = document.getElementById("contacto-id");
const contactoNombres = document.getElementById("contacto-nombres");
const contactoApellidos = document.getElementById("contacto-apellidos");
const contactoEmail = document.getElementById("contacto-email");
const contactoCelular = document.getElementById("contacto-celular");
const contactoNumDoc = document.getElementById("contacto-num-doc");
const contactoOrigen = document.getElementById("contacto-origen");
const tabDatos = document.getElementById("tab-datos");
const tabInteracciones = document.getElementById("tab-interacciones");
const tabDatosBtn = document.getElementById("tab-datos-btn");
const tabInteraccionesBtn = document.getElementById("tab-interacciones-btn");
const tabCotizaciones = document.getElementById("tab-cotizaciones");
const tabCotizacionesBtn = document.getElementById("tab-cotizaciones-btn");
const tabStatus = document.getElementById("tab-status");
const tabStatusBtn = document.getElementById("tab-status-btn");
const interaccionesContainer = document.getElementById("interacciones-container");
const intCanal = document.getElementById("int-canal");
const intResumen = document.getElementById("int-resumen");
const intSeguimiento = document.getElementById("int-seguimiento");
const intProximo = document.getElementById("int-proximo");
const btnAddInteraccion = document.getElementById("btn-add-interaccion");
const cotArchivo = document.getElementById("cot-archivo");
const cotMonto = document.getElementById("cot-monto");
const cotObs = document.getElementById("cot-observaciones");
const btnAddCotizacion = document.getElementById("btn-add-cotizacion");
const cotizacionesContainer = document.getElementById("cotizaciones-container");
const statusContainer = document.getElementById("status-container");
const columnToggles = document.querySelectorAll(".req-col-toggle");
const columnState = {};

const monthNames = [
  "Enero",
  "Febrero",
  "Marzo",
  "Abril",
  "Mayo",
  "Junio",
  "Julio",
  "Agosto",
  "Septiembre",
  "Octubre",
  "Noviembre",
  "Diciembre",
];

const channelLabels = {
  email: "Email",
  whatsapp: "WhatsApp",
  telefono: "Telefono",
  "reunion presencial": "Reunion presencial",
  "reunion virtual": "Reunion virtual",
  otro: "Otro",
};

function normalizeChannel(value) {
  return String(value || "").trim().toLowerCase();
}

function getDisplayUser(row) {
  const currentName = currentUser?.user ? String(currentUser.user).toLowerCase() : "";
  const rowName = row?.usuario ? String(row.usuario).toLowerCase() : "";
  if (currentName && rowName && currentName === rowName) return "Tú";
  const fullName = `${row?.nombres ?? ""} ${row?.apellidos ?? ""}`.trim();
  if (fullName) return fullName;
  return row?.usuario || "-";
}

function parseLimaTimestamp(value) {
  const raw = String(value || "").trim();
  if (!raw) return null;
  const normalized = raw.replace(" ", "T");
  const parts = normalized.split("T");
  if (!parts[0]) return null;
  const dateParts = parts[0].split("-").map(Number);
  if (dateParts.length < 3) return null;
  const [y, m, d] = dateParts;
  let hh = 0, mm = 0, ss = 0;
  if (parts[1]) {
    const timeParts = parts[1].split(":").map(Number);
    hh = timeParts[0] ?? 0;
    mm = timeParts[1] ?? 0;
    ss = timeParts[2] ?? 0;
  }
  if (!y || !m || !d) return null;
  return Date.UTC(y, m - 1, d, hh + 5, mm, ss);
}

function getLimaNowTimestamp() {
  const dtf = new Intl.DateTimeFormat("en-CA", {
    timeZone: "America/Lima",
    year: "numeric",
    month: "2-digit",
    day: "2-digit",
    hour: "2-digit",
    minute: "2-digit",
    second: "2-digit",
    hour12: false,
  });
  const parts = dtf.formatToParts(new Date());
  const get = (type) => Number(parts.find((p) => p.type === type)?.value || 0);
  const y = get("year");
  const m = get("month");
  const d = get("day");
  const hh = get("hour");
  const mm = get("minute");
  const ss = get("second");
  return Date.UTC(y, m - 1, d, hh + 5, mm, ss);
}

function escapeHtml(value) {
  return String(value ?? "")
    .replace(/&/g, "&amp;")
    .replace(/</g, "&lt;")
    .replace(/>/g, "&gt;")
    .replace(/"/g, "&quot;")
    .replace(/'/g, "&#39;");
}

function truncateText(value, limit) {
  const text = String(value ?? "");
  if (!limit || text.length <= limit) return text;
  return `${text.slice(0, limit).trim()}...`;
}

function safeUrl(raw) {
  if (!raw) return "";
  const trimmed = String(raw).trim();
  if (!trimmed) return "";
  if (trimmed.startsWith("http://") || trimmed.startsWith("https://")) return trimmed;
  if (trimmed.startsWith("assets/") || trimmed.startsWith("/")) return `${base}${trimmed.replace(/^\/+/, "")}`;
  return "";
}

function setTab(tab) {
  const isDatos = tab === "datos";
  const isInteracciones = tab === "interacciones";
  const isCotizaciones = tab === "cotizaciones";
  const isStatus = tab === "status";
  tabDatos?.classList.toggle("hidden", !isDatos);
  tabInteracciones?.classList.toggle("hidden", !isInteracciones);
  tabCotizaciones?.classList.toggle("hidden", !isCotizaciones);
  tabStatus?.classList.toggle("hidden", !isStatus);
  if (tabDatosBtn) {
    tabDatosBtn.classList.toggle("bg-primary", isDatos);
    tabDatosBtn.classList.toggle("text-white", isDatos);
    tabDatosBtn.classList.toggle("bg-gray-200", !isDatos);
  }
  if (tabInteraccionesBtn) {
    tabInteraccionesBtn.classList.toggle("bg-primary", isInteracciones);
    tabInteraccionesBtn.classList.toggle("text-white", isInteracciones);
    tabInteraccionesBtn.classList.toggle("bg-gray-200", !isInteracciones);
  }
  if (tabCotizacionesBtn) {
    tabCotizacionesBtn.classList.toggle("bg-primary", isCotizaciones);
    tabCotizacionesBtn.classList.toggle("text-white", isCotizaciones);
    tabCotizacionesBtn.classList.toggle("bg-gray-200", !isCotizaciones);
  }
  if (tabStatusBtn) {
    tabStatusBtn.classList.toggle("bg-primary", isStatus);
    tabStatusBtn.classList.toggle("text-white", isStatus);
    tabStatusBtn.classList.toggle("bg-gray-200", !isStatus);
  }
}

function applyColumnVisibility() {
  const table = document.getElementById("tablaReq");
  if (!table) return;
  table.querySelectorAll("[data-col]").forEach((el) => {
    const col = el.getAttribute("data-col");
    const isVisible = columnState[col] !== false;
    el.classList.toggle("hidden", !isVisible);
  });
}

const openModal = () => {
  modal?.classList.remove("hidden");
  modal?.classList.add("flex");
};
const closeModal = () => {
  modal?.classList.add("hidden");
  modal?.classList.remove("flex");
};
const openEmpresaQuick = () => {
  empresaQuickModal?.classList.remove("hidden");
  empresaQuickModal?.classList.add("flex");
};
const closeEmpresaQuick = () => {
  empresaQuickModal?.classList.add("hidden");
  empresaQuickModal?.classList.remove("flex");
};
const clearEmpresaQuick = () => {
  if (empresaRuc) empresaRuc.value = "";
  if (empresaRazon) empresaRazon.value = "";
  if (empresaNombre) empresaNombre.value = "";
  if (empresaOrigen) empresaOrigen.value = "";
};
const openContactoQuick = () => {
  contactoQuickModal?.classList.remove("hidden");
  contactoQuickModal?.classList.add("flex");
};
const closeContactoQuick = () => {
  contactoQuickModal?.classList.add("hidden");
  contactoQuickModal?.classList.remove("flex");
};

const formatMoney = (val) => {
  if (val === null || val === undefined || val === "") return "";
  return Number(val).toLocaleString("es-PE", { minimumFractionDigits: 2, maximumFractionDigits: 2 });
};

function formatDatePretty(dateStr) {
  if (!dateStr) return "";
  const normalized = dateStr.replace(" ", "T");
  const d = new Date(normalized);
  if (Number.isNaN(d.getTime())) return dateStr;
  const day = String(d.getDate()).padStart(2, "0");
  const month = monthNames[d.getMonth()] || "";
  const year = d.getFullYear();
  return `${day} ${month} ${year}`;
}

async function fetchJson(url, options = {}) {
  const resp = await fetch(url, {
    headers: { "Content-Type": "application/json" },
    credentials: "same-origin",
    ...options,
  });
  const text = await resp.text();
  try {
    return JSON.parse(text);
  } catch {
    const start = text.indexOf("{"),
      end = text.lastIndexOf("}");
    if (start !== -1 && end !== -1 && end > start) {
      try {
        return JSON.parse(text.slice(start, end + 1));
      } catch {}
    }
    throw new Error("JSON parse error");
  }
}

async function fetchJsonForm(url, formData) {
  const resp = await fetch(url, { method: "POST", body: formData, credentials: "same-origin" });
  const text = await resp.text();
  try {
    return JSON.parse(text);
  } catch {
    const start = text.indexOf("{"),
      end = text.lastIndexOf("}");
    if (start !== -1 && end !== -1 && end > start) {
      try {
        return JSON.parse(text.slice(start, end + 1));
      } catch {}
    }
    throw new Error("JSON parse error");
  }
}

async function loadRequerimientos() {
  const q = searchInput.value.trim();
  const url = `${apiUrl}?action=list&page=${paginacion.currentPage}&limit=${paginacion.rowsPerPage}&q=${encodeURIComponent(
    q
  )}`;
  const data = await fetchJson(url);
  if (!data.success) {
    tableBody.innerHTML = `<tr><td class="px-2 py-1 text-gray-500 text-xs" colspan="26">${escapeHtml(data.message || "No se pudo procesar la solicitud.")}</td></tr>`;
    paginacion.updatePaginationInfo(0);
    paginacion.renderPaginationControls();
    return;
  }
  state.total = data.total || 0;
  if (totalLabel) totalLabel.textContent = `Total: ${state.total}`;
  renderTable(data.data || []);
  paginacion.updatePaginationInfo(state.total);
  paginacion.renderPaginationControls();
}

function renderTable(rows) {
  tableBody.innerHTML = "";
  rows.forEach((r) => {
    const tr = document.createElement("tr");
    tr.className = "border-b hover:bg-slate-50";
    const empresaNombre =
      r.empresa_nombre || r.empresa_nombre_comercial || r.empresa_razon_social || r.empresa_id || "";
    const empresaRuc = r.empresa_ruc ?? "";
    const contactoNombre = r.contacto_nombre || `${r.contacto_nombres ?? ""} ${r.contacto_apellidos ?? ""}`.trim();
    const contactoCargo = r.contacto_cargo ?? "";
    const contactoTelefono = r.contacto_telefono ?? "";
    const contactoEmail = r.contacto_email ?? "";
    const tipoEmpresa = r.empresa_tipo_empresa ?? "";
    const tipoCondicion = r.tipo_condicion_proyecto ?? "";
    const motivoCompra = r.motivo_compra ?? "";
    const formaPago = r.forma_pago ?? "";
    const nivelTentativa = r.nivel_tentativa ?? "";
    const categoriaProductos = r.categoria_productos ?? "";
    const usuarioApoyo = `${r.usuario_apoyo_nombres ?? ""} ${r.usuario_apoyo_apellidos ?? ""}`.trim();
    const usuarioNombre = `${r.usuario_nombres ?? ""} ${r.usuario_apellidos ?? ""}`.trim();
    const fechaReq = formatDatePretty(r.fecha_requerida);
    const fechaUpd = formatDatePretty(r.historia_updated_at || r.updated_at);
    const empresaDisplay = truncateText(empresaNombre, 30);
    const empresaTitle = escapeHtml(empresaNombre);
    const descripcionRaw = r.descripcion ?? "";
    const descripcionDisplay = truncateText(descripcionRaw, 50);
    tr.innerHTML = `
    <td class="px-2 py-1">${escapeHtml(r.id)}</td>
    <td class="px-2 py-1">${escapeHtml(r.titulo ?? "")}</td>
    <td class="px-2 py-1 max-w-[240px] truncate" title="${escapeHtml(descripcionRaw)}">${escapeHtml(descripcionDisplay)}</td>
    <td class="px-2 py-1" data-col="ruc">${escapeHtml(empresaRuc)}</td>
    <td class="px-2 py-1 max-w-[200px] truncate" title="${empresaTitle}">${escapeHtml(empresaDisplay)}</td>
    <td class="px-2 py-1" data-col="tipo_empresa">${escapeHtml(tipoEmpresa)}</td>
    <td class="px-2 py-1">${escapeHtml(contactoNombre)}</td>
    <td class="px-2 py-1" data-col="cargo">${escapeHtml(contactoCargo)}</td>
    <td class="px-2 py-1" data-col="telefono">${escapeHtml(contactoTelefono)}</td>
    <td class="px-2 py-1" data-col="email">${escapeHtml(contactoEmail)}</td>
    <td class="px-2 py-1">${escapeHtml(r.origen ?? "")}</td>
    <td class="px-2 py-1">${escapeHtml(r.prioridad ?? "")}</td>
    <td class="px-2 py-1" data-col="tipo_condicion_proyecto">${escapeHtml(tipoCondicion)}</td>
    <td class="px-2 py-1" data-col="motivo_compra">${escapeHtml(motivoCompra)}</td>
    <td class="px-2 py-1" data-col="forma_pago">${escapeHtml(formaPago)}</td>
    <td class="px-2 py-1" data-col="nivel_tentativa">${escapeHtml(nivelTentativa)}</td>
    <td class="px-2 py-1" data-col="categoria_productos">${escapeHtml(categoriaProductos)}</td>
    <td class="px-2 py-1" data-col="usuario_apoyo">${escapeHtml(usuarioApoyo)}</td>
    <td class="px-2 py-1">${escapeHtml(r.etapa ?? "")}</td>
    <td class="px-2 py-1">${escapeHtml(r.lead_id ?? "")}</td>
    <td class="px-2 py-1">${escapeHtml(formatMoney(r.monto_estimado))}</td>
    <td class="px-2 py-1">${escapeHtml(formatMoney(r.monto_real))}</td>
    <td class="px-2 py-1">${escapeHtml(fechaReq)}</td>
    <td class="px-2 py-1">${escapeHtml(fechaUpd)}</td>
    <td class="px-2 py-1">${escapeHtml(usuarioNombre)}</td>
    <td class="px-2 py-1 space-x-1">
      <button class="btn-edit bg-yellow-500 text-white text-xs px-2 py-1 rounded mr-1" data-id="${r.id}"><i class="fas fa-pen"></i></button>
      <button class="btn-delete-row bg-red-600 text-white text-xs px-2 py-1 rounded" data-id="${r.id}"><i class="fas fa-trash"></i></button>
    </td>
    `;
    tableBody.appendChild(tr);
  });
  applyColumnVisibility();
}

function resetForm() {
  formReq.reset();
  state.editingId = null;
  document.getElementById("req-id").value = "";
  if (btnSave) btnSave.textContent = "Guardar";
  setTab("datos");
  if (modalTitle) modalTitle.textContent = "Nuevo Requerimiento";
  if (contactoSelect) contactoSelect.innerHTML = `<option value="">Seleccione contacto</option>`;
  if (empresaIdField) empresaIdField.value = "";
  if (empresaInput) empresaInput.value = "";
  clearEmpresaQuick();
  if (contactoEmpresaNombre) contactoEmpresaNombre.value = "";
  if (contactoEmpresaId) contactoEmpresaId.value = "";
  if (contactoId) contactoId.value = "";
  if (contactoNombres) contactoNombres.value = "";
  if (contactoApellidos) contactoApellidos.value = "";
  if (contactoEmail) contactoEmail.value = "";
  if (contactoCelular) contactoCelular.value = "";
  if (contactoNumDoc) contactoNumDoc.value = "";
  if (contactoOrigen) contactoOrigen.value = "";
  const origenSel = document.getElementById("req-origen");
  if (origenSel) origenSel.value = "";
  const prioSel = document.getElementById("req-prioridad");
  if (prioSel) prioSel.value = "media";
  const tipoCond = document.getElementById("req-tipo-condicion");
  if (tipoCond) tipoCond.value = "Estandar";
  const motivoCompra = document.getElementById("req-motivo-compra");
  if (motivoCompra) motivoCompra.value = "";
  const formaPago = document.getElementById("req-forma-pago");
  if (formaPago) formaPago.value = "";
  const nivelTentativa = document.getElementById("req-nivel-tentativa");
  if (nivelTentativa) nivelTentativa.value = "medio";
  const categoria = document.getElementById("req-categoria");
  if (categoria) categoria.value = "";
  if (usuarioApoyoSelect) usuarioApoyoSelect.innerHTML = `<option value="">Seleccione</option>`;
  loadUsuariosApoyo();
  if (interaccionesContainer)
    interaccionesContainer.innerHTML = '<div class="text-gray-500 text-xs">Sin interacciones.</div>';
  if (cotizacionesContainer)
    cotizacionesContainer.innerHTML = '<div class="text-gray-500 text-xs">Sin cotizaciones.</div>';
  if (statusContainer)
    statusContainer.innerHTML = '<div class="text-gray-500 text-xs">Sin historial.</div>';
  if (cotArchivo) cotArchivo.value = "";
  if (cotMonto) cotMonto.value = "";
  if (cotObs) cotObs.value = "";
}

async function searchEmpresas(term) {
  const url = `${base}controller/empresa/empresa.php?action=list&q=${encodeURIComponent(term)}&limit=10&page=1`;
  const resp = await fetch(url, { credentials: "same-origin" });
  const data = await resp.json();
  if (!data.success) return [];
  return data.data || [];
}

function renderEmpresaDropdown(items) {
  if (!empresaDropdown) return;
  empresaDropdown.innerHTML = "";
  if (!items.length) {
    empresaDropdown.classList.add("hidden");
    return;
  }
  items.forEach((e) => {
    const div = document.createElement("div");
    div.className = "px-2 py-1 hover:bg-gray-100 cursor-pointer text-sm";
    const nombre = e.nombre_comercial || e.razon_social || `Empresa #${e.id}`;
    const ruc = e.ruc ? e.ruc : "";
    div.dataset.id = e.id;
    div.dataset.text = nombre;
    div.innerHTML = `<div class="font-medium">${escapeHtml(nombre)}</div><div class="text-xs text-gray-600">${escapeHtml(
      ruc
    )}</div>`;
    empresaDropdown.appendChild(div);
  });
  empresaDropdown.classList.remove("hidden");
}

async function loadContactosByEmpresa(empresaId, selectedId = "") {
  if (!contactoSelect) return;
  contactoSelect.innerHTML = `<option value="">Seleccione contacto</option>`;
  if (!empresaId) return;
  const url = `${base}controller/empresa/empresa.php?action=list_contactos&empresa_id=${empresaId}`;
  const resp = await fetch(url, { credentials: "same-origin" });
  const data = await resp.json();
  if (!data.success) return;
  (data.data || []).forEach((c) => {
    const opt = document.createElement("option");
    opt.value = c.id;
    opt.textContent = `${c.nombres ?? ""} ${c.apellidos ?? ""}`.trim() || `Contacto #${c.id}`;
    if (String(selectedId) === String(c.id)) opt.selected = true;
    contactoSelect.appendChild(opt);
  });
}

async function loadUsuariosApoyo(selectedId = "") {
  if (!usuarioApoyoSelect) return;
  usuarioApoyoSelect.innerHTML = `<option value="">Seleccione</option>`;
  const url = `${base}controller/usuarios/listar.php?page=1&filtro=`;
  const resp = await fetch(url, { credentials: "same-origin" });
  const data = await resp.json();
  if (!data.success) return;
  (data.data || []).forEach((u) => {
    const opt = document.createElement("option");
    opt.value = u.idusuario ?? u.id ?? "";
    const nombre = `${u.nombres ?? ""} ${u.apellidos ?? ""}`.trim();
    opt.textContent = nombre || u.usuario || `Usuario #${opt.value}`;
    if (String(selectedId) === String(opt.value)) opt.selected = true;
    usuarioApoyoSelect.appendChild(opt);
  });
}

async function handleSave(event) {
  event.preventDefault();
  const formData = new FormData(formReq);
  const payload = Object.fromEntries(formData.entries());
  [
    "lead_id",
    "contacto_id",
    "monto_estimado",
    "fecha_requerida",
    "tipo_condicion_proyecto",
    "motivo_compra",
    "forma_pago",
    "nivel_tentativa",
    "categoria_productos",
    "usuario_apoyo_id",
  ].forEach((k) => {
    if (payload[k] === "") payload[k] = null;
  });
  if (!payload.empresa_id) return Swal.fire({ icon: "warning", text: "Seleccione una empresa" });
  if (!payload.contacto_id) return Swal.fire({ icon: "warning", text: "Seleccione un contacto" });
  if (!payload.origen) return Swal.fire({ icon: "warning", text: "Seleccione un origen" });
  if (payload.prioridad === "") payload.prioridad = null;
  const isEdit = Boolean(payload.id);
  try {
    const resp = await fetchJson(`${apiUrl}?action=${isEdit ? "update" : "create"}`, {
      method: "POST",
      body: JSON.stringify(payload),
    });
    if (!resp.success) {
      Swal.fire({ icon: "error", text: resp.message || "No se pudo guardar" });
      return;
    }
    Swal.fire({ icon: "success", title: "Guardado", text: resp.message || "Listo", timer: 1500, showConfirmButton: false });
    resetForm();
    closeModal();
    loadRequerimientos();
  } catch {
    Swal.fire({ icon: "error", text: "Error al guardar" });
  }
}

async function handleDelete() {
  if (!state.editingId) return;
  const confirmDel = await Swal.fire({
    icon: "warning",
    title: "Eliminar requerimiento",
    text: "Desea eliminar este registro?",
    showCancelButton: true,
    confirmButtonText: "Si, eliminar",
  });
  if (!confirmDel.isConfirmed) return;
  try {
    const resp = await fetchJson(`${apiUrl}?action=delete`, { method: "POST", body: JSON.stringify({ id: state.editingId }) });
    if (!resp.success) {
      Swal.fire({ icon: "error", text: resp.message || "No se pudo eliminar" });
      return;
    }
    Swal.fire({ icon: "success", title: "Eliminado", timer: 1200, showConfirmButton: false });
    resetForm();
    closeModal();
    loadRequerimientos();
  } catch {
    Swal.fire({ icon: "error", text: "Error al eliminar" });
  }
}

async function handleEdit(id) {
  try {
    const data = await fetchJson(`${apiUrl}?action=get&id=${id}`);
    if (!data.success) {
      Swal.fire({ icon: "error", text: data.message || "No se pudo cargar" });
      return;
    }
    const r = data.data;
    state.editingId = r.id;
    document.getElementById("req-id").value = r.id;
    empresaIdField.value = r.empresa_id ?? "";
    const nombreComercial = (r.empresa_nombre_comercial ?? "").trim();
    const razonSocial = (r.empresa_razon_social ?? "").trim();
    const empresaTexto =
      nombreComercial && nombreComercial !== "-" ? nombreComercial : razonSocial || r.empresa_id || "";
    empresaInput.value = empresaTexto;
    await loadContactosByEmpresa(r.empresa_id, r.contacto_id);
    document.getElementById("req-contacto").value = r.contacto_id ?? "";
    document.getElementById("req-titulo").value = r.titulo ?? "";
    document.getElementById("req-descripcion").value = r.descripcion ?? "";
    document.getElementById("req-monto").value = r.monto_estimado ?? "";
    document.getElementById("req-fecha").value = r.fecha_requerida ?? "";
    document.getElementById("req-etapa").value = r.etapa ?? "Pendiente Atender";
    document.getElementById("req-origen").value = r.origen ?? "";
    document.getElementById("req-prioridad").value = r.prioridad ?? "";
    document.getElementById("req-tipo-condicion").value = r.tipo_condicion_proyecto ?? "";
    document.getElementById("req-motivo-compra").value = r.motivo_compra ?? "";
    document.getElementById("req-forma-pago").value = r.forma_pago ?? "";
    document.getElementById("req-nivel-tentativa").value = r.nivel_tentativa ?? "";
    document.getElementById("req-categoria").value = r.categoria_productos ?? "";
    await loadUsuariosApoyo(r.usuario_apoyo_id ?? "");
    if (btnSave) btnSave.textContent = "Actualizar";
    if (modalTitle) modalTitle.textContent = `Editar Requerimiento #${r.id}`;
    setTab("datos");
    await loadInteracciones(r.id);
    await loadCotizaciones(r.id);
    await loadHistorialStatus(r.contacto_id);
    openModal();
  } catch {
    Swal.fire({ icon: "error", text: "Error al cargar el requerimiento" });
  }
}

btnRefresh?.addEventListener("click", () => {
  paginacion.currentPage = 1;
  loadRequerimientos();
});
if (columnToggles.length) {
  columnToggles.forEach((cb) => {
    const col = cb.getAttribute("data-col");
    if (!col) return;
    columnState[col] = cb.checked;
    cb.addEventListener("change", () => {
      columnState[col] = cb.checked;
      applyColumnVisibility();
    });
  });
  applyColumnVisibility();
}
if (pageSizeSel) {
  pageSizeSel.addEventListener("change", () => {
    updateRowsPerPage("req-pageSize");
    loadRequerimientos();
  });
  paginacion.rowsPerPage = parseInt(pageSizeSel.value || "25", 10);
}
paginacion.renderFunction = () => loadRequerimientos();

formReq?.addEventListener("submit", handleSave);
btnOpenModal?.addEventListener("click", () => {
  resetForm();
  openModal();
});
btnCloseModal?.addEventListener("click", closeModal);
modal?.addEventListener("click", (e) => {
  if (e.target === modal) closeModal();
});
btnEmpresaNuevo?.addEventListener("click", () => {
  clearEmpresaQuick();
  openEmpresaQuick();
});
btnEmpresaQuickClose?.addEventListener("click", closeEmpresaQuick);
btnEmpresaQuickCancel?.addEventListener("click", closeEmpresaQuick);
empresaQuickModal?.addEventListener("click", (e) => {
  if (e.target === empresaQuickModal) closeEmpresaQuick();
});
btnContactoNuevo?.addEventListener("click", () => {
  if (!empresaIdField?.value) {
    Swal.fire({ icon: "warning", text: "Seleccione o cree una empresa primero." });
    return;
  }
  if (contactoQuickTitle) contactoQuickTitle.textContent = "Nuevo Contacto";
  if (contactoEmpresaNombre) contactoEmpresaNombre.value = empresaInput?.value || "";
  if (contactoEmpresaId) contactoEmpresaId.value = empresaIdField.value || "";
  if (contactoId) contactoId.value = "";
  openContactoQuick();
});
btnContactoQuickClose?.addEventListener("click", closeContactoQuick);
btnContactoQuickCancel?.addEventListener("click", closeContactoQuick);
contactoQuickModal?.addEventListener("click", (e) => {
  if (e.target === contactoQuickModal) closeContactoQuick();
});
tabDatosBtn?.addEventListener("click", () => setTab("datos"));
tabInteraccionesBtn?.addEventListener("click", () => setTab("interacciones"));
tabCotizacionesBtn?.addEventListener("click", () => setTab("cotizaciones"));
tabStatusBtn?.addEventListener("click", () => setTab("status"));

tableBody?.addEventListener("click", (e) => {
  const target = e.target;
  if (target.closest(".btn-edit")) {
    const id = target.closest("button")?.dataset?.id;
    if (id) handleEdit(id);
    return;
  }
  if (target.closest(".btn-delete-row")) {
    const id = target.closest("button")?.dataset?.id;
    if (id) {
      state.editingId = id;
      handleDelete();
    }
  }
});

let empresaSearchTimeout;
function triggerEmpresaSearch(term) {
  clearTimeout(empresaSearchTimeout);
  empresaSearchTimeout = setTimeout(async () => {
    const rows = await searchEmpresas(term);
    renderEmpresaDropdown(rows);
  }, 120);
}
empresaInput?.addEventListener("input", (e) => {
  const term = e.target.value.trim();
  if (!term) {
    empresaIdField.value = "";
    triggerEmpresaSearch("");
    return;
  }
  triggerEmpresaSearch(term);
});
empresaInput?.addEventListener("focus", () => {
  triggerEmpresaSearch(empresaInput.value.trim());
});
empresaDropdown?.addEventListener("click", async (e) => {
  const item = e.target.closest("div[data-id]");
  if (!item) return;
  const id = item.dataset.id;
  const text = item.dataset.text;
  empresaIdField.value = id;
  empresaInput.value = text;
  renderEmpresaDropdown([]);
  await loadContactosByEmpresa(id);
});
empresaInput?.addEventListener("blur", () => {
  setTimeout(() => renderEmpresaDropdown([]), 150);
});
empresaInput?.addEventListener("change", () => {
  if (!empresaInput.value.trim()) {
    empresaIdField.value = "";
    contactoSelect.innerHTML = `<option value="">Seleccione contacto</option>`;
  }
});
contactoSelect?.addEventListener("change", () => {
  if (state.editingId) {
    loadHistorialStatus(contactoSelect.value);
  }
});

function ensureMinProximo() {
  if (!intProximo) return;
  const now = new Date();
  intProximo.min = now.toISOString().slice(0, 16);
}
intProximo?.addEventListener("change", () => {
  ensureMinProximo();
  if (intProximo.value && intSeguimiento) intSeguimiento.checked = true;
});
intProximo?.addEventListener("focus", ensureMinProximo);

async function loadInteracciones(reqId) {
  if (!interaccionesContainer) return;
  interaccionesContainer.innerHTML = '<div class="text-xs text-gray-500">Cargando...</div>';
  const data = await fetchJson(`${apiUrl}?action=list_interacciones&requerimiento_id=${reqId}`);
  if (!data.success) {
    interaccionesContainer.innerHTML = `<div class="text-xs text-red-600">${data.message || "No se pudieron cargar las interacciones"}</div>`;
    return;
  }
  const rows = data.data || [];
  if (!rows.length) {
    interaccionesContainer.innerHTML = '<div class="text-gray-500 text-xs">Sin interacciones.</div>';
    return;
  }
  rows.sort((a, b) => {
    const aTs = new Date((a.updated_at || a.created_at || "").replace(" ", "T")).getTime();
    const bTs = new Date((b.updated_at || b.created_at || "").replace(" ", "T")).getTime();
    return bTs - aTs;
  });
  const nowLimaTs = getLimaNowTimestamp();
  const frag = document.createDocumentFragment();
  rows.forEach((r, idx) => {
    const div = document.createElement("div");
    const isLatest = idx === 0;
    const requiere = Number(r.requiere_seguimiento) === 1;
    const proximoTs = parseLimaTimestamp(r.proximo_seguimiento);
    const isOverdue = proximoTs !== null && proximoTs < nowLimaTs;
    let baseClass = "timeline-card card-normal";
    if (isOverdue) {
      baseClass = "timeline-card card-overdue";
    } else if (requiere) {
      baseClass = "timeline-card card-followup";
    }
    div.className = isLatest ? `${baseClass} card-latest` : baseClass;
    const canalRaw = r.canal || "Sin canal";
    const canalKey = normalizeChannel(canalRaw);
    const canalLabel = channelLabels[canalKey] || canalRaw;
    const resumen = r.resumen || "";
    const resumenText = resumen.replace(/\s+/g, " ").trim();
    const limit = 70;
    const isLong = resumenText.length > limit;
    const previewText = isLong ? `${resumenText.slice(0, limit).trimEnd()}...` : resumenText;
    const fecha = formatDatePretty(r.updated_at || r.created_at);
    const proximo = formatDatePretty(r.proximo_seguimiento);
    const user = getDisplayUser(r);
    const seguimiento = requiere ? "Si" : "No";
    const proximoTxt = requiere ? proximo || "no definido" : null;
    const proximoClass = isOverdue ? "followup-overdue" : isLatest && proximo ? "followup-important" : "";
    const followupBoxClass = isOverdue ? "follow-up-overdue" : "";
    const toggleButton = isLong
      ? '<button class="btn-link" type="button" data-toggle="collapseText">Ver mas</button>'
      : "";
    div.innerHTML = `
      <div class="timeline-header">
        <span class="channel-pill">${escapeHtml(canalLabel)}</span>
        <span>${escapeHtml(fecha)}</span>
      </div>
      <div class="comment" data-collapsible data-limit="70" data-expanded="0">
        <span class="comment-preview">${escapeHtml(previewText)}</span>
        ${isLong ? `<span class="comment-full">${escapeHtml(resumenText)}</span>` : ""}
      </div>
      ${requiere ? `<div class="follow-up-alert ${followupBoxClass}">
        <div class="left"><strong>Requiere seguimiento:</strong> ${escapeHtml(seguimiento)}</div>
        <div class="right ${proximoClass}"><strong>Proximo seguimiento:</strong> ${escapeHtml(proximoTxt || "-")}</div>
      </div>` : ""}
      <div class="card-footer">
        <span class="user-tag"> ${escapeHtml(user || "-")}</span>
        ${toggleButton}
      </div>
    `;
    frag.appendChild(div);
  });
  interaccionesContainer.innerHTML = "";
  interaccionesContainer.appendChild(frag);
}

interaccionesContainer?.addEventListener("click", (event) => {
  const btn = event.target.closest('[data-toggle="collapseText"]');
  if (!btn) return;
  const card = btn.closest(".timeline-card");
  if (!card) return;
  const el = card.querySelector('[data-collapsible]');
  if (!el) return;
  const preview = el.querySelector(".comment-preview");
  const full = el.querySelector(".comment-full");
  if (!preview || !full) return;
  const expanded = el.getAttribute("data-expanded") === "1";
  if (!expanded) {
    preview.style.display = "none";
    full.style.display = "inline";
    el.setAttribute("data-expanded", "1");
    btn.textContent = "Ver menos";
  } else {
    full.style.display = "none";
    preview.style.display = "inline";
    el.setAttribute("data-expanded", "0");
    btn.textContent = "Ver mas";
  }
});

async function loadCotizaciones(reqId) {
  if (!cotizacionesContainer) return;
  cotizacionesContainer.innerHTML = '<div class="text-xs text-gray-500">Cargando...</div>';
  const data = await fetchJson(`${apiUrl}?action=list_cotizaciones&requerimiento_id=${reqId}`);
  if (!data.success) {
    cotizacionesContainer.innerHTML = `<div class="text-xs text-red-600">${data.message || "No se pudieron cargar las cotizaciones"}</div>`;
    return;
  }
  const rows = data.data || [];
  if (!rows.length) {
    cotizacionesContainer.innerHTML = '<div class="text-gray-500 text-xs">Sin cotizaciones.</div>';
    return;
  }
  rows.sort((a, b) => {
    const aTs = new Date((a.updated_at || a.created_at || "").replace(" ", "T")).getTime();
    const bTs = new Date((b.updated_at || b.created_at || "").replace(" ", "T")).getTime();
    return bTs - aTs;
  });
  const frag = document.createDocumentFragment();
  rows.forEach((r, idx) => {
    const div = document.createElement("div");
    div.className = idx === 0 ? "q-card q-normal q-latest" : "q-card q-normal";
    const fecha = formatDatePretty(r.created_at);
    const monto = r.monto ? formatMoney(r.monto) : "";
    const obs = r.observaciones || "";
    const obsText = obs.replace(/\s+/g, " ").trim();
    const limit = 70;
    const isLong = obsText.length > limit;
    const previewText = isLong ? `${obsText.slice(0, limit).trimEnd()}...` : obsText;
    const user = getDisplayUser(r);
    const archivoRaw = r.archivo || "";
    const archivoUrl = safeUrl(archivoRaw);
    const archivoHtml = archivoUrl
      ? `<a href="${escapeHtml(archivoUrl)}" target="_blank" rel="noopener" class="q-link">Ver PDF</a>`
      : '<span class="text-xs text-gray-500">Sin archivo</span>';
    const toggleButton = isLong
      ? '<button class="btn-link" type="button" data-toggle="collapseText">Ver mas</button>'
      : "";
    div.innerHTML = `
      <div class="q-head">
        <div>
          <div class="q-title">Cotizacion</div>
          ${archivoHtml}
        </div>
        <span class="q-date">${escapeHtml(fecha)}</span>
      </div>
      <div class="q-amount">
        <span class="q-currency">Monto</span>
        <strong>${escapeHtml(monto || "-")}</strong>
      </div>
      <div class="q-note" data-collapsible data-limit="70" data-expanded="0">
        <span class="note-preview">${escapeHtml(previewText || "-")}</span>
        ${isLong ? `<span class="note-full">${escapeHtml(obsText)}</span>` : ""}
      </div>
      <div class="q-foot">
        <span class="q-user">Usuario: <strong>${escapeHtml(user)}</strong></span>
        ${toggleButton}
      </div>
    `;
    frag.appendChild(div);
  });
  cotizacionesContainer.innerHTML = "";
  cotizacionesContainer.appendChild(frag);
}

async function loadHistorialStatus(contactoId) {
  if (!statusContainer) return;
  if (!contactoId) {
    statusContainer.innerHTML = '<div class="text-gray-500 text-xs">Sin historial.</div>';
    return;
  }
  statusContainer.innerHTML = '<div class="text-xs text-gray-500">Cargando...</div>';
  const data = await fetchJson(`${apiUrl}?action=list_historial_status&contacto_id=${contactoId}`);
  if (!data.success) {
    statusContainer.innerHTML = `<div class="text-xs text-red-600">${escapeHtml(data.message || "No se pudo cargar el historial")}</div>`;
    return;
  }
  const rows = data.data || [];
  if (!rows.length) {
    statusContainer.innerHTML = '<div class="text-gray-500 text-xs">Sin historial.</div>';
    return;
  }
  const frag = document.createDocumentFragment();
  rows.forEach((r) => {
    const div = document.createElement("div");
    div.className = "border border-gray-200 rounded p-2 bg-white";
    const fecha = formatDatePretty(r.fechaingreso);
    const user = `${r.nombres ?? ""} ${r.apellidos ?? ""}`.trim() || r.usuario || r.idusuario || "-";
    const status = r.status || "";
    div.innerHTML = `
      <div class="text-xs text-gray-500">${escapeHtml(fecha)}</div>
      <div class="text-sm text-gray-800 mt-1">${escapeHtml(status)}</div>
      <div class="text-xs text-gray-600 mt-1">Usuario: ${escapeHtml(user)}</div>
    `;
    frag.appendChild(div);
  });
  statusContainer.innerHTML = "";
  statusContainer.appendChild(frag);
}

cotizacionesContainer?.addEventListener("click", (event) => {
  const btn = event.target.closest('[data-toggle="collapseText"]');
  if (!btn) return;
  const card = btn.closest(".q-card");
  if (!card) return;
  const el = card.querySelector('[data-collapsible]');
  if (!el) return;
  const preview = el.querySelector(".note-preview");
  const full = el.querySelector(".note-full");
  if (!preview || !full) return;
  const expanded = el.getAttribute("data-expanded") === "1";
  if (!expanded) {
    preview.style.display = "none";
    full.style.display = "inline";
    el.setAttribute("data-expanded", "1");
    btn.textContent = "Ver menos";
  } else {
    full.style.display = "none";
    preview.style.display = "inline";
    el.setAttribute("data-expanded", "0");
    btn.textContent = "Ver mas";
  }
});

async function handleAddInteraccion(event) {
  event?.preventDefault?.();
  if (!state.editingId) {
    Swal.fire({ icon: "warning", text: "Guarda el requerimiento antes de agregar interacciones." });
    setTab("datos");
    return;
  }
  ensureMinProximo();
  const requiere = (intSeguimiento?.checked ? 1 : 0) || (intProximo?.value ? 1 : 0);
  const payload = {
    requerimiento_id: state.editingId,
    canal: intCanal?.value || null,
    resumen: intResumen?.value || null,
    requiere_seguimiento: requiere,
    proximo_seguimiento: intProximo?.value || null,
  };
  try {
    const resp = await fetchJson(`${apiUrl}?action=add_interaccion`, { method: "POST", body: JSON.stringify(payload) });
    if (!resp.success) {
      Swal.fire({ icon: "error", text: resp.message || "No se pudo guardar la interaccion" });
      return;
    }
    Swal.fire({ icon: "success", title: "Guardado", timer: 1200, showConfirmButton: false });
    if (intCanal) intCanal.value = "";
    if (intResumen) intResumen.value = "";
    if (intSeguimiento) intSeguimiento.checked = false;
    if (intProximo) intProximo.value = "";
    await loadInteracciones(state.editingId);
    setTab("interacciones");
  } catch {
    Swal.fire({ icon: "error", text: "Error al guardar la interaccion" });
  }
}
btnAddInteraccion?.addEventListener("click", handleAddInteraccion);

async function handleAddCotizacion(event) {
  event?.preventDefault?.();
  if (!state.editingId) {
    Swal.fire({ icon: "warning", text: "Guarda el requerimiento antes de agregar cotizaciones." });
    setTab("datos");
    return;
  }
  const file = cotArchivo?.files?.[0];
  if (!file) {
    Swal.fire({ icon: "warning", text: "Selecciona el PDF de la cotizacion." });
    return;
  }
  const montoVal = (cotMonto?.value || "").trim();
  if (!montoVal) {
    Swal.fire({ icon: "warning", text: "Ingresa el monto de la cotizacion." });
    return;
  }
  const formData = new FormData();
  formData.append("requerimiento_id", state.editingId);
  formData.append("monto", montoVal);
  formData.append("observaciones", cotObs?.value || "");
  formData.append("archivo", file);
  try {
    const resp = await fetchJsonForm(`${apiUrl}?action=add_cotizacion`, formData);
    if (!resp.success) {
      Swal.fire({ icon: "error", text: resp.message || "No se pudo guardar la cotizacion" });
      return;
    }
    Swal.fire({ icon: "success", title: "Guardado", timer: 1200, showConfirmButton: false });
    if (cotArchivo) cotArchivo.value = "";
    if (cotMonto) cotMonto.value = "";
    if (cotObs) cotObs.value = "";
    await loadCotizaciones(state.editingId);
    setTab("cotizaciones");
    loadRequerimientos();
  } catch {
    Swal.fire({ icon: "error", text: "Error al guardar la cotizacion" });
  }
}
btnAddCotizacion?.addEventListener("click", handleAddCotizacion);

async function saveEmpresaQuick() {
  const ruc = empresaRuc?.value?.trim() || "";
  const razon = empresaRazon?.value?.trim() || "";
  const nombre = empresaNombre?.value?.trim() || "";
  const origen = empresaOrigen?.value?.trim() || "";
  if (!ruc || !razon) {
    Swal.fire({ icon: "warning", text: "RUC y Razon social son obligatorios" });
    return;
  }
  const payload = { ruc, razon_social: razon, nombre_comercial: nombre, origen };
  try {
    const resp = await fetchJson(`${base}controller/empresa/empresa.php?action=create`, {
      method: "POST",
      body: JSON.stringify(payload),
    });
    if (!resp.success) {
      Swal.fire({ icon: "error", text: resp.message || "No se pudo crear la empresa" });
      return;
    }
    const newId = resp.id;
    empresaIdField.value = newId;
    empresaInput.value = nombre || razon;
    await loadContactosByEmpresa(newId);
    closeEmpresaQuick();
    Swal.fire({ icon: "success", title: "Empresa creada", timer: 1200, showConfirmButton: false });
  } catch {
    Swal.fire({ icon: "error", text: "Error al crear la empresa" });
  }
}
btnEmpresaQuickSave?.addEventListener("click", saveEmpresaQuick);

btnContactoQuickSave?.addEventListener("click", async () => {
  const empresaId = contactoEmpresaId?.value || empresaIdField?.value;
  if (!empresaId) {
    Swal.fire({ icon: "warning", text: "Seleccione o cree una empresa primero." });
    return;
  }
  const nombres = contactoNombres?.value?.trim() || "";
  const apellidos = contactoApellidos?.value?.trim() || "";
  const email = contactoEmail?.value?.trim() || "";
  const celular = contactoCelular?.value?.trim() || "";
  const numDoc = contactoNumDoc?.value?.trim() || "";
  const origen = contactoOrigen?.value?.trim() || "";
  if (!nombres) {
    Swal.fire({ icon: "warning", text: "Nombres es obligatorio" });
    return;
  }
  if (!origen) {
    Swal.fire({ icon: "warning", text: "Origen es obligatorio" });
    return;
  }
  const payload = {
    empresa_id: empresaId,
    nombres,
    apellidos,
    email,
    celular,
    num_doc: numDoc,
    origen,
  };
  try {
    const resp = await fetchJson(`${base}controller/empresa/empresa.php?action=create_contacto`, {
      method: "POST",
      body: JSON.stringify(payload),
    });
    if (!resp.success) {
      Swal.fire({ icon: "error", text: resp.message || "No se pudo crear el contacto" });
      return;
    }
    await loadContactosByEmpresa(empresaId, resp.id);
    closeContactoQuick();
    if (contactoNombres) contactoNombres.value = "";
    if (contactoApellidos) contactoApellidos.value = "";
    if (contactoEmail) contactoEmail.value = "";
    if (contactoCelular) contactoCelular.value = "";
    if (contactoNumDoc) contactoNumDoc.value = "";
    if (contactoOrigen) contactoOrigen.value = "";
    Swal.fire({ icon: "success", title: "Guardado", timer: 1200, showConfirmButton: false });
  } catch {
    Swal.fire({ icon: "error", text: "Error al crear el contacto" });
  }
});

loadRequerimientos().catch((err) => console.error(err));
